#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# @File    : 缓存处理.py
# @Time    : 2022/12/13 14:26
# @Author  : bruxelles_li
# @Software: PyCharm

"""
基于内存缓存
使用 memory_cache 实例即可
"""
import shutil
import time
import logging

logging.basicConfig(level=logging.INFO, format='%(asctime)s [%(levelname)s] <%(processName)s> (%(threadName)s) %('
                                               'message)s')
logger = logging.getLogger(__name__)


class Value:
    def __init__(self, value, put_time, expired):
        """
        缓存值对象

        :param value: 具体的值
        :param put_time: 放入缓存的时间
        :param expired: 缓存失效时间
        """
        self.value = value
        self.put_time = put_time
        self.expired = expired

    def __str__(self):
        return f"value: {self.value}  put_time: {self.put_time}  expired: {self.expired}"


class MemoryCache:

    def __init__(self):
        self.__cache = {}

    def set_value(self, k, v, expired):
        """
        将值放入缓存中

        :param k: 缓存的 key
        :param v: 缓存值
        :param expired: 缓存失效时间，单位秒(s)
        """
        current_timestamp = int(time.time())  # 获取当前时间戳 10 位 秒级
        value = Value(v, current_timestamp, expired)
        self.__cache[k] = value
        logger.info("已放入缓存, {}: {}".format(k, value))

    def check_key(self, k):
        """
        检查缓存是否可用

        :param k: 缓存 key
        :return: True or False
        """
        current_timestamp = int(time.time())
        value = self.__cache.get(k, None)
        # 考虑k不存在的情况
        if value is None:
            return False
        differ = current_timestamp - value.put_time
        if differ > value.expired:
            del self.__cache[k]  # 证明缓存失效了，删除键值对
            logger.info("缓存已失效, key: {}".format(k))
            return False
        return True

    def get_value(self, k):
        """
        通过缓存key获取值

        :param k: key
        :return: value
        """
        if self.check_key(k):
            return self.__cache[k].value
        return None


memory_cache = MemoryCache()


if __name__ == "__main__":
    import json, os
    # memory_cache.set_value('my_blog', 'sunnyc.icu', 3)  # 设置一个 3 秒过期的键值对
    # memory_cache.set_value('my_github', 'hczs', 20)  # 设置一个 6 秒过期的键值对
    dic_result = {
        "my_blog": 'sunnyc.icu',
        "my_github": "hczs"

    }
    json_result = json.dumps(dic_result)
    path = "测试文件/test.json"
    with open(path, 'w', encoding='utf-8') as file:
        file.write(json_result)

    # os.remove(path)
    if os.path.isfile(path):
        with open("测试文件/test.json", 'r', encoding='utf-8') as f:
            dict_result = json.load(f)
        print(dict_result)
    else:
        print("no")




    # time.sleep(5)
    # a = "my_blog"
    # if a == "my_blog":
    #     print(memory_cache.get_value(a))
    # else:
    #     print('my_github: ', memory_cache.get_value('my_github'))

