package com.zzsn.excelexportservice.service;

import com.alibaba.cloud.commons.lang.StringUtils;
import com.zzsn.excelexportservice.dto.*;
import com.zzsn.excelexportservice.exportStrategys.ExportStrategy;
import com.zzsn.excelexportservice.factorys.ExportStrategyFactory;

import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;

import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.*;
import com.zzsn.excelexportservice.dto.ExportDataResponse.Header;

/**
 * @Author: lxj
 * @Date: 2025/9/18 11:07
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class ExcelExportService {

    static {
        // 设置无头模式，避免字体配置问题
        System.setProperty("java.awt.headless", "true");
    }

    @Resource
    private CmsSearch cmsSearch;

    private final ExportStrategyFactory exportStrategyFactory;

    public void export(List<ExportReq> exportReqList, HttpServletRequest request, HttpServletResponse response) {
        if (exportReqList == null || exportReqList.isEmpty()) {
            sendError(response, "导出失败: 入参为空");
            return;
        }

        SXSSFWorkbook workbook = null;
        try {
            workbook = new SXSSFWorkbook(1000);
            workbook.setCompressTempFiles(true);
            int i=1;
            for (ExportReq exportReq : exportReqList) {
                ExportDataResponse resp = null;
                String serviceName = exportReq.getServiceName();
                Map<String, Object> queryParams = exportReq.getQueryParams() != null
                        ? exportReq.getQueryParams()
                        : new HashMap<>();

                ExportStrategy strategy = exportStrategyFactory.getStrategy(serviceName, exportReq.getApiPath());
                if (strategy == null) {
                    log.error("No strategy found for service: {}, path: {}", serviceName, exportReq.getApiPath());
                    sendError(response, "导出失败: 未找到匹配的服务策略");
                    return;
                }

                resp = strategy.execute(request, queryParams, exportReq);

                if (resp == null || (resp.getCode() != 0 && resp.getCode() != 200)) {
                    log.error("导出数据失败, serviceName: {}, apiPath: {}, errorMsg: {}",
                            serviceName, exportReq.getApiPath(), resp != null ? resp.getMsg() : "null");
                    sendError(response, resp != null ? resp.getMsg() : "导出失败: 未知错误");
                    return;
                }

                // 安全创建 sheet
                String sheetName = generateSafeSheetName(
                        StringUtils.isNotBlank(exportReq.getSheetName())
                                ? exportReq.getSheetName()
                                : "Sheet" + i++
                );
                Sheet sheet = workbook.createSheet(sheetName);

                int startRow = 0;
                int endCol = 13;

                // 判断是否有图片
                if (StringUtils.isNotBlank(exportReq.getImagesBase64())) {
                    log.info("开始插入图片------->" + System.currentTimeMillis());
                    drawImg(workbook, sheet, exportReq.getImagesBase64(), startRow, endCol);
                    startRow++; // 图片占用一行，从下一行开始写标题
                }

                List<Map<String, Object>> dataList = resp.getDataList() != null
                        ? resp.getDataList()
                        : new ArrayList<>();
                List<Header> headers = resp.getHeaders() != null
                        ? resp.getHeaders()
                        : new ArrayList<>();

                // 写标题
                log.info("开始插入标题------->" + System.currentTimeMillis());
                handlerExcelTitle(workbook, sheet, startRow, headers);

                // 写数据
                log.info("开始写入数据------->" + System.currentTimeMillis());
                handlerExcelData(workbook, sheet, startRow + 1, headers, dataList);

                // 设置手动列宽，避免自动列宽计算导致的字体问题
                setupManualColumnWidth(sheet, headers.size());
            }

            // 最后写到 response
            log.info("生成excel结束------->" + System.currentTimeMillis());
            writeExcelToResponse("导出", workbook, response);

        } catch (Exception e) {
            log.error("导出Excel异常", e);
            sendError(response, "导出失败: " + e.getMessage());
        } finally {
            // 确保资源释放
            if (workbook != null) {
                try {
                    workbook.close();
                } catch (IOException e) {
                    log.warn("关闭workbook异常", e);
                }
                workbook.dispose();
            }
        }
    }

    /**
     * 生成安全的sheet名称
     */
    private String generateSafeSheetName(String name) {
        if (StringUtils.isBlank(name)) {
            return "Sheet1";
        }
        // 移除Excel不允许的字符：\ / * [ ] : ?
        String safeName = name.replaceAll("[\\\\/*\\[\\]:?]", "");
        // 限制长度在31个字符以内
        return safeName.length() > 31 ? safeName.substring(0, 31) : safeName;
    }

    /**
     * 设置手动列宽
     */
    private void setupManualColumnWidth(Sheet sheet, int columnCount) {
        try {
            // 根据列数设置合适的宽度，避免自动列宽计算
            for (int i = 0; i < columnCount; i++) {
                sheet.setColumnWidth(i, 5000); // 设置固定宽度
            }
        } catch (Exception e) {
            log.warn("设置列宽失败，使用默认宽度", e);
        }
    }

    @SuppressWarnings("unchecked")
    private <T> T getParam(Map<String, Object> params, String key, T defaultVal) {
        if (params == null) {
            return defaultVal;
        }
        Object val = params.get(key);
        if (val == null) {
            return defaultVal;
        }
        try {
            return (T) val;
        } catch (ClassCastException e) {
            log.warn("参数 {} 类型不正确，期望 {}, 实际 {}", key, defaultVal.getClass(), val.getClass());
            return defaultVal;
        }
    }

    private static void sendError(HttpServletResponse response, String msg) {
        try {
            response.reset();
            response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            response.setContentType("text/plain;charset=UTF-8");
            response.getWriter().write(msg != null ? msg : "未知错误");
        } catch (IOException e) {
            log.error("发送错误响应失败", e);
        }
    }

    // 插入图片（Base64）
    private void drawImg(SXSSFWorkbook workbook, Sheet sheet, String base64Img, int rowIndex, int endCol) {
        try {
            Row firstRow = sheet.createRow(rowIndex);
            for (int i = 0; i < endCol; i++) {
                firstRow.createCell(i);
            }
            firstRow.setHeightInPoints(230);
            // 合并单元格
            sheet.addMergedRegion(new CellRangeAddress(rowIndex, rowIndex, 0, endCol));

            if (StringUtils.isNotBlank(base64Img)) {
                int start = base64Img.indexOf(",");
                String imgData = (start != -1) ? base64Img.substring(start + 1) : base64Img;

                if (StringUtils.isNotBlank(imgData)) {
                    byte[] imgBytes = Base64.getDecoder().decode(imgData);

                    Drawing<?> drawingPatriarch = sheet.createDrawingPatriarch();
                    ClientAnchor anchor = drawingPatriarch.createAnchor(0, 0, 0, 0, 0, rowIndex, endCol, rowIndex + 1);
                    int pictureIdx = workbook.addPicture(imgBytes, Workbook.PICTURE_TYPE_PNG);
                    drawingPatriarch.createPicture(anchor, pictureIdx);
                }
            }
        } catch (Exception e) {
            log.warn("插入图片失败", e);
        }
    }

    // 写入标题
    private void handlerExcelTitle(Workbook workbook, Sheet sheet, int rowIndex, List<Header> headers) {
        if (headers == null || headers.isEmpty()) {
            return;
        }

        try {
            Row row = sheet.createRow(rowIndex);
            CellStyle titleStyle = getTitleStyle(workbook);
            for (int i = 0; i < headers.size(); i++) {
                Cell cell = row.createCell(i);
                cell.setCellStyle(titleStyle);
                String title = headers.get(i) != null ? headers.get(i).getTitle() : "";
                cell.setCellValue(title != null ? title : "");
            }
        } catch (Exception e) {
            log.warn("写入标题失败", e);
        }
    }

    // Title 样式
    private CellStyle getTitleStyle(Workbook workbook) {
        CellStyle style = workbook.createCellStyle();
        try {
            Font font = workbook.createFont();
            font.setFontName("Arial"); // 使用通用字体
            font.setBold(true);
            style.setFont(font);
        } catch (Exception e) {
            log.warn("创建标题样式失败，使用默认样式", e);
        }
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        return style;
    }

    // 写入数据
    private void handlerExcelData(Workbook workbook, Sheet sheet, int startRowIndex,
                                  List<Header> headers, List<Map<String, Object>> dataList) {
        if (headers == null || headers.isEmpty() || dataList == null || dataList.isEmpty()) {
            return;
        }

        try {
            CellStyle dataStyle = getDataStyle(workbook);
            for (int j = 0; j < dataList.size(); j++) {
                Row row = sheet.createRow(startRowIndex + j);
                Map<String, Object> dataMap = dataList.get(j);
                if (dataMap == null) {
                    continue;
                }

                for (int i = 0; i < headers.size(); i++) {
                    Cell cell = row.createCell(i);
                    cell.setCellStyle(dataStyle);

                    Header header = headers.get(i);
                    if (header == null) {
                        cell.setCellValue("");
                        continue;
                    }

                    String field = header.getField();
                    Object value = dataMap.get(field);
                    cell.setCellValue(value == null ? "" : value.toString());
                }
            }
        } catch (Exception e) {
            log.warn("写入数据失败", e);
        }
    }

    // 数据样式
    private CellStyle getDataStyle(Workbook workbook) {
        CellStyle style = workbook.createCellStyle();
        try {
            Font font = workbook.createFont();
            font.setFontName("Arial"); // 使用通用字体
            style.setFont(font);
        } catch (Exception e) {
            log.warn("创建数据样式失败，使用默认样式", e);
        }
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        return style;
    }

    // 将数据写入 HttpServletResponse
    private void writeExcelToResponse(String fileNamePrefix, SXSSFWorkbook workbook, HttpServletResponse response) {
        try {
            response.reset();
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            String fileName = URLEncoder.encode(
                    (fileNamePrefix != null ? fileNamePrefix : "导出") + System.currentTimeMillis() + ".xlsx",
                    "UTF-8"
            );
            response.setHeader("Content-Disposition", "attachment;filename=" + fileName);
            response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");

            workbook.write(response.getOutputStream());
            response.getOutputStream().flush();
        } catch (IOException e) {
            log.error("写入响应流失败", e);
            sendError(response, "导出 Excel 异常：" + e.getMessage());
        }
    }
}