package com.zzsn.excelexportservice.exportStrategys;


import com.zzsn.excelexportservice.dto.CountryReq;
import com.zzsn.excelexportservice.dto.ExportDataResponse;
import com.zzsn.excelexportservice.dto.ExportReq;
import com.zzsn.excelexportservice.feign.InternationalFeign;
import jakarta.annotation.PostConstruct;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;


import java.util.*;


/**
 * @Author: lxj
 * @Date: 2025/9/25 16:52
 */
@Component
public class InternationalStrategy implements ExportStrategy {

    @Autowired
    private InternationalFeign internationalFeign;

    private final Map<String, TriFunction<HttpServletRequest, Map<String, Object>, ExportReq, ExportDataResponse<?>>> pathMapping = new HashMap<>();


    @PostConstruct
    public void init() {
        List<String> postApiPaths = Arrays.asList(
                "/international/country/merchandise/export/excel",
                "/international/country/getTradeServ/export/excel",
                "/international/country/getTradeServ/category/export/excel",
                "/international/country/getTradeIndices/export/excel",
                "/international/country/getEconomic/export/excel",
                "/international/country/getEconomic/Expenditure/export/excel",
                "/international/country/getFdiAndExternal/export/excel",
                "/international/country/getFdiAndExternal/capitalFlow/export/excel",
                "/international/country/getGoodsAndServ/export/excel",
                "/international/country/getGoodsAndServ/openness/export/excel",
                "/international/country/getLivingArea/export/excel",
                "/international/country/getOther/export/excel",
                "/international/country/getOther/economy/export/excel"
        );

        for (String path : postApiPaths) {
            pathMapping.put(path, (request, queryParams, exportReq) -> {
                try {
                    CountryReq req = new CountryReq((String) queryParams.get("country"));
                    switch (path) {
                        case "/international/country/merchandise/export/excel":
                            return internationalFeign.merchandiseExport(req);
                        case "/international/country/getTradeServ/export/excel":
                            return internationalFeign.getTradeServExport(req);
                        case "/international/country/getTradeServ/category/export/excel":
                            return internationalFeign.getTradeServCategoryExport(req);
                        case "/international/country/getTradeIndices/export/excel":
                            return internationalFeign.getTradeIndicesExport(req);
                        case "/international/country/getEconomic/export/excel":
                            return internationalFeign.getEconomicExport(req);
                        case "/international/country/getEconomic/Expenditure/export/excel":
                            return internationalFeign.getEconomicExpenditureExport(req);
                        case "/international/country/getFdiAndExternal/export/excel":
                            return internationalFeign.getFdiAndExternal(req);
                        case "/international/country/getFdiAndExternal/capitalFlow/export/excel":
                            return internationalFeign.getFdiAndExternalCapitalFlowExport(req);
                        case "/international/country/getGoodsAndServ/export/excel":
                            return internationalFeign.getGoodsAndServExport(req);
                        case "/international/country/getGoodsAndServ/openness/export/excel":
                            return internationalFeign.getGoodsAndServOpennessExport(req);
                        case "/international/country/getLivingArea/export/excel":
                            return internationalFeign.getLivingAreaExport(req);
                        case "/international/country/getOther/export/excel":
                            return internationalFeign.getOtherExport(req);
                        case "/international/country/getOther/economy/export/excel":
                            return internationalFeign.getOtherEconomyExport(req);
                        default:
                            return new ExportDataResponse<>(404, "未找到匹配的接口", Collections.emptyList(), Collections.emptyList());
                    }
                } catch (Exception e) {
                    // 不抛 500，返回业务错误码 2，前端可以根据 msg 显示提示
                    return new ExportDataResponse<>(2, "调用国际接口失败：" + e.getMessage(), Collections.emptyList(), Collections.emptyList());
                }
            });
        }

        // 特殊接口：@RequestParam String country
        pathMapping.put("/international/country/generalInfo/export/excel",
                (request, queryParams, exportReq) -> {
                    try {
                        String country = (String) queryParams.get("country");
                        return internationalFeign.getCountryInfoExport(country);
                    } catch (Exception e) {
                        return new ExportDataResponse<>(2, "调用国际化接口失败：" + e.getMessage(), Collections.emptyList(), Collections.emptyList());
                    }
                });
    }
    @Override
    public String getServiceName() {
        return "international-operation";
    }

    /**
     * 返回 null，表示所有 apiPath 由本类处理
     */
    @Override
    public String getApiPath() {
        return null;
    }

    @Override
    public ExportDataResponse<?> execute(HttpServletRequest request, Map<String, Object> queryParams, ExportReq exportReq) {
        TriFunction<HttpServletRequest, Map<String, Object>, ExportReq, ExportDataResponse<?>> func =
                pathMapping.get(exportReq.getApiPath());
        if (func == null) {
            throw new IllegalArgumentException("No mapping for apiPath: " + exportReq.getApiPath());
        }
        return func.apply(request, queryParams, exportReq);
    }
}

