#!/bin/bash
# build.sh - 功能包括：镜像构建，推送镜像，处理none镜像。
# author: lixingyu
# 使用方法:
#   $ ./build.sh <env_name> <build_timestamp>
# 参数:
#   env_name:环境变量 build_timestamp:构建时间戳

set -eu

# 获取脚本的绝对路径（包括文件名）
script_path=$(readlink -f "$0")
# 获取脚本所在的目录的绝对路径
script_dir=$(dirname "$script_path")

# 函数定义
function print_usage {
  # 打印使用说明
  sed -n '2,7p' "$0"
}

function handle {
  local env_name=$1
  local build_timestamp=$2

  echo "执行的环境变量: ${env_name}"
  echo "执行的构建时间戳: ${build_timestamp}"
  if [ ! -f "$script_dir/env/env.${env_name}" ]; then
    echo "错误: 文件 'env.${env_name}' 不存在." >&2
    exit 1
  fi
  . $script_dir/env/env.${env_name}

  local username="${repo_username}"
  local passwd="${repo_passwd}"
  local name="${svc_name}-${svc_env}"
  local version=${version}-${build_timestamp}
  local internal_ip="${internal_ip}"

  echo "构建名称: ${name}"
  echo "构建端口: ${svc_port}"
  echo "部署节点: ${internal_ip}"
  echo "构建版本: ${version}"
  echo "推送仓库: ${domain}/${namespace}"
  echo "完整镜像: ${domain}/${namespace}/${name}:${version}"

  echo "---构建镜像---"
  docker build --build-arg NACOS_REGISTER_IP=${internal_ip} --build-arg NACOS_SERVER=${nacos_server} --build-arg NACOS_NAMESPACE=${nacos_namespace} \
  --build-arg ENV_NAME=${svc_env} --build-arg PORT=${svc_port} -f ./build/Dockerfile -t ${domain}/${namespace}/${name}:${version} .

  echo "---推送镜像---"
  docker login -u=$username -p=${passwd} ${domain}
  docker push ${domain}/${namespace}/${name}:${version}

  echo "---清理none镜像---"
  docker image prune -af
}

# 主程序入口点
function main {
  if [ "$#" -ne 2 ]; then
    print_usage
    echo "错误: 需要提供两个个参数 <env_name> <build_timestamp>" >&2
    exit 1
  fi

  local env_name=$1
  local build_timestamp=$2

  handle "${env_name}" "${build_timestamp}"
}

# 错误处理
trap 'echo 发生了错误，脚本中断.' ERR

# 调用主函数
main "$@"
