package com.zzsn.thinktank.controller;

import cn.hutool.poi.excel.ExcelReader;
import cn.hutool.poi.excel.ExcelUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.thinktank.entity.ThinktankBasicInfo;
import com.zzsn.thinktank.service.LeaderCategoryService;
import com.zzsn.thinktank.service.ThinktankBasicInfoService;
import com.zzsn.thinktank.util.AccessTokenUtil;
import com.zzsn.thinktank.vo.Result;
import com.zzsn.thinktank.vo.ThinktankBasicInfoListVo;
import com.zzsn.thinktank.vo.ThinktankBasicInfoVo;
import com.zzsn.thinktank.vo.ThinktankInfoVo;
import com.zzsn.thinktank.util.ExcelExportUtil;
import com.zzsn.thinktank.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/4/24 10:28
 * @Content:
 */
@Slf4j
@RestController
@RequestMapping("/api/basicinfo")
public class ThinktankBasicInfoController {

    @Autowired
    ThinktankBasicInfoService thinktankBasicInfoService;
    @Autowired
    LeaderCategoryService leaderCategoryService;


    /**
     * 新增
     *
     * @param thinktankBasicInfo
     * @return
     */
    @PostMapping("/add")
    public Result<?> add(@RequestBody ThinktankBasicInfo thinktankBasicInfo) {
        log.info("新增信息：{}", thinktankBasicInfo);
        return thinktankBasicInfoService.add(thinktankBasicInfo);
    }

    /**
     * 根据id主键删除
     *
     * @param id
     * @return
     */
    @GetMapping("/del")
    public Result<?> del(@RequestParam(name = "id", required = true) String id) {
        log.info("根据id删除:{}", id);
        return thinktankBasicInfoService.del(id);
    }

    /**
     * 根据id修改信息
     *
     * @param thinktankBasicInfo
     * @return
     */
    @PostMapping("/edit")
    public Result<?> edit(@RequestBody ThinktankBasicInfo thinktankBasicInfo) {
        log.info("根据id修改信息:{}", thinktankBasicInfo);
        if (thinktankBasicInfo.getId() == null)
            return Result.error(400, "id不能为空");
        return thinktankBasicInfoService.edit(thinktankBasicInfo);
    }

    /**
     * 根据id主键查询
     *
     * @param id
     * @return
     */
    @GetMapping("getById")
    public Result<?> getById(@RequestParam(name = "id", required = true) String id) {
        log.info("根据id主键查询:{}", id);
        ThinktankBasicInfoVo thinktankBasicInfo = thinktankBasicInfoService.getInfoById(id);
        return Result.OK(thinktankBasicInfo);
    }

    /**
     * 根据id主键查询智库信息
     * @param id
     * @return
     */
    @GetMapping("info")
    public Result<?> getThinktankBasicInfoById(@RequestParam(name="id",required=true) String id, HttpServletRequest request) {
        String userId = "";
        try {
            userId = AccessTokenUtil.getUserIdFromAccessToken(request);
        } catch (Exception e) {
            if (userId == null || "".equals(userId)) {
                userId = request.getSession().getId();
            }
        }
        ThinktankInfoVo thinktankInfoVo = thinktankBasicInfoService.getThinktankBasicInfoById(id,userId);
        return Result.OK(thinktankInfoVo);
    }

    /**
     * 收藏/取消收藏
     * @param id
     * @return
     */
    @GetMapping("collect")
    public Result<?> collect(@RequestParam(name="id",required=true) String id, HttpServletRequest request) {
        String userId = "";
        try {
            userId = AccessTokenUtil.getUserIdFromAccessToken(request);
        } catch (Exception e) {
            if (userId == null || "".equals(userId)) {
                userId = request.getSession().getId();
            }
        }
        thinktankBasicInfoService.collect(id,userId);
        return Result.OK("操作成功");
    }

    /**
     * 智库关联ott
     * @param baseCode 智库编码
     * @param ottCode ott主键
     * @return
     */
    @GetMapping("relation")
    public Result<?> relation(String baseCode, String ottCode) {
        thinktankBasicInfoService.relation(baseCode,ottCode);
        return Result.OK("操作成功");
    }

    /**
     * 已收藏智库列表
     * @return
     */
    @GetMapping("collectList")
    public Result<?> collectList(HttpServletRequest request) {
        String userId = "";
        try {
            userId = AccessTokenUtil.getUserIdFromAccessToken(request);
        } catch (Exception e) {
            if (userId == null || "".equals(userId)) {
                userId = request.getSession().getId();
            }
        }
        List<ThinktankBasicInfo> thinktankBasicInfos = thinktankBasicInfoService.collectList(userId);
        return Result.OK(thinktankBasicInfos);
    }


    @PostMapping("/list")
    public Result<?> getList(@RequestBody ThinktankBasicInfoListVo thinktankBasicInfoListVo) {
        log.info("智库信息条件分页查询:{}", thinktankBasicInfoListVo);
        Integer pageNo = thinktankBasicInfoListVo.getPageNo();
        Integer pageSize = thinktankBasicInfoListVo.getPageSize();
        if (pageNo < 1)
            pageNo = 1;
        if (pageSize < 0)
            pageSize = 10;
        Integer office = pageSize * (pageNo - 1);
        thinktankBasicInfoListVo.setOffset(office);
        return thinktankBasicInfoService.getList(thinktankBasicInfoListVo);
    }


    /**
     * 智库列表（yjzx）
     */
    @PostMapping("/queryPageList")
    public Result<?> queryPageList(@RequestBody(required = false) ThinktankBasicInfoListVo vo,
                                   @RequestParam(name="pageNo", defaultValue="1") Integer pageNo,
                                   @RequestParam(name="pageSize", defaultValue="10") Integer pageSize,
                                   HttpServletRequest req) throws Exception {
        IPage<ThinktankBasicInfo> pageList = thinktankBasicInfoService.pageList(pageNo,pageSize,vo,req);
        return Result.OK(pageList);
    }


    /**
     * 智库区域/国别/领域统计分析（yjzx）
     * @param type(continent按洲   country按国家  tag按领域)
     */
    @GetMapping("/queryStatisticalAnalysis")
    public Result<?> queryStatisticalAnalysis(String type,HttpServletRequest req) throws Exception {
        ThinktankBasicInfoListVo vo = new ThinktankBasicInfoListVo();
        Page<ThinktankBasicInfo> page = new Page<ThinktankBasicInfo>(1, 999999);
        List<Map<String,Object>> mapList = thinktankBasicInfoService.queryStatisticalAnalysis(type,page,vo,req);
        return Result.OK(mapList);
    }

    /**
     * 模板下载
     *
     * @param response
     */
    @GetMapping("/downloadTemplate")
    public void downloadTemplate(HttpServletResponse response) {
        thinktankBasicInfoService.downloadTemplate(response);
    }


    /**
     * 批量导入数据
     *
     * @param file excel文件
     * @author lkg
     * @date 2024/8/22
     */
    @PostMapping("/batchImport")
    public Result<?> batchImport(@RequestParam("file") MultipartFile file) {
        int index = Objects.requireNonNull(file.getOriginalFilename()).lastIndexOf(".");
        String fileSuffix = file.getOriginalFilename().substring(index + 1);
        if ("xls".equals(fileSuffix) || "xlsx".equals(fileSuffix)) {
            boolean flag = checkExcel(file);
            if (!flag) {
                return Result.error("模版有误，请使用正确的模板！");
            }
        } else {
            return Result.error("不支持的文件类型！");
        }
        try {
            ExcelReader reader = ExcelUtil.getReader(file.getInputStream());
            Map<String, String> header = new HashMap<>();
            header.put("智库机构编码", "codeId");
            header.put("* 中文全称", "chineseWhole");
            header.put("中文简称", "chineseSimple");
            header.put("英文全称", "englishWhole");
            header.put("英文简称", "englishSimple");
            header.put("官网", "officialWebsite");
            header.put("* 所属国家", "belongCountry");
            header.put("语言", "lang");
            header.put("运营状态", "operateStatus");
            header.put("营利性质", "profitNature");
            header.put("智库性质", "nature");
            header.put("研究领域", "tagName");
            header.put("影响力和地位", "influencePosition");
            header.put("人员规模", "staffSize");
            header.put("智库规模", "scale");
            header.put("成果类型", "achievementType");
            header.put("资金来源", "fundsSource");
            header.put("原文全称", "originalWhole");
            header.put("原文简称", "originalSimple");
            header.put("简介", "biographicalNotes");
            header.put("成立时间", "establishTime");
            header.put("是否收费(1-是;0-否)", "charge");
            header.put("地址", "address");
            header.put("创办单位/所属单位", "belongUnit");
            reader.setHeaderAlias(header);
            List<ThinkTankFileVO> thinkTankImportVOList = reader.read(1, 2, ThinkTankFileVO.class);
            if (CollectionUtils.isNotEmpty(thinkTankImportVOList)) {
                boolean requiredFlag = true;
                for (ThinkTankFileVO thinkTankImportVO : thinkTankImportVOList) {
                    if (StringUtils.isEmpty(thinkTankImportVO.getChineseWhole()) || StringUtils.isEmpty(thinkTankImportVO.getBelongCountry())) {
                        requiredFlag = false;
                        break;
                    }
                }
                if (!requiredFlag) {
                    return Result.error("必填项【中文全称和所属国家】有缺失,请核对后再导入。");
                }
                String msg = thinktankBasicInfoService.batchImport(thinkTankImportVOList);
                return Result.OK(msg);
            } else {
                return Result.error("文件数据为空。");
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return Result.OK();
    }

    @PostMapping("/batchExport")
    public byte[] batchExport(@RequestBody ThinktankBasicInfoListVo one) {
        byte[] bytes = null;
        try {
            List<ThinktankBasicInfo> thinkTankBasicInfos = thinktankBasicInfoService.exportListByCondition(one.getId(), one.getCodeId(),
                    one.getTypeId(), one.getThinktankName(), one.getTagId(), one.getKeyword());
            if (CollectionUtils.isNotEmpty(thinkTankBasicInfos)) {
                String[] arr = new String[]{"智库机构id", "智库机构编码", "中文全称", "中文简称", "英文全称", "英文简称", "官网", "所属国家", "语言", "经营状态","营利性质",
                        "智库性质", "研究领域", "影响力和地位", "人员规模","智库规模", "成果类型", "资金来源", "原文全称", "原文简称", "简介", "成立时间", "是否收费(1-是;0-否)",
                        "地址", "创办单位/所属单位"};
                XSSFWorkbook workbook = new XSSFWorkbook();
                //基本信息
                List<List<String>> rows = new ArrayList<>();
                thinkTankBasicInfos.forEach(each -> rows.add(each.toList()));
                ExcelExportUtil.exportExcelData(workbook, 0, Arrays.asList(arr), rows, "智库基本信息");
                //统计信息
                String[] header = new String[]{"智库机构id", "智库机构编码", "中文全称", "关联信息源数量", "信息采集总量", "关联信息源编码",
                        "关联信息源名称", "栏目名称", "网址", "启用状态(1-启用;0-禁用)","信息源采集数量", "信息源近一月采集量"};
                List<ThinkTankSourceVO> thinkTankSourceVOS = thinktankBasicInfoService.thinkTankCollectCount(thinkTankBasicInfos);
                List<List<String>> rowList = new ArrayList<>();
                thinkTankSourceVOS.forEach(e->rowList.add(e.toList()));
                ExcelExportUtil.exportExcelData(workbook, 1, Arrays.asList(header), rowList, "智库采集统计信息");
                ByteArrayOutputStream outStream = new ByteArrayOutputStream();
                workbook.write(outStream);
                // 将字节流转换为InputStream
                bytes = outStream.toByteArray();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return bytes;
    }

    @GetMapping("/listBaseData")
    public Result<?> getListBaseData(@RequestParam(name = "id", required = true) String id,
                                     @RequestParam(name = "pageNo", required = true) Integer pageNo,
                                     @RequestParam(name = "pageSize", required = true) Integer pageSize) {
        log.info("智库成果分页查询:{}", id);
        if (pageNo < 1)
            pageNo = 1;
        if (pageSize < 0)
            pageSize = 10;
        Integer office = pageSize * (pageNo - 1);
        return thinktankBasicInfoService.getListBaseData(id, pageNo, pageSize, office);
    }

    private boolean checkExcel(MultipartFile multipartFile) {
        Workbook workbook = ExcelExportUtil.getWorkbook(multipartFile);
        if (workbook != null) {
            List<String> list = ExcelExportUtil.getSheetTitles(workbook);
            return list != null
                    && list.contains("智库机构编码")
                    && list.contains("* 中文全称")
                    && list.contains("中文简称")
                    && list.contains("英文全称")
                    && list.contains("英文简称")
                    && list.contains("官网")
                    && list.contains("* 所属国家")
                    && list.contains("语言")
                    && list.contains("营利性质")
                    && list.contains("智库性质")
                    && list.contains("研究领域")
                    && list.contains("影响力和地位")
                    && list.contains("智库规模")
                    && list.contains("成果类型")
                    && list.contains("资金来源")
                    && list.contains("原文全称")
                    && list.contains("原文简称")
                    && list.contains("简介")
                    && list.contains("成立时间")
                    && list.contains("是否收费(1-是;0-否)")
                    && list.contains("地址")
                    && list.contains("创办单位/所属单位");
        }
        return false;
    }

    /**
     * 智库维护
     * @return
     */
    @PostMapping("importExcel")
    public Result think(MultipartFile file) {
        thinktankBasicInfoService.thinkTank(file);
        return Result.OK();
    }

}
