#!/bin/bash
# run.sh - 功能包括：运行容器，处理none镜像。
# author: lixingyu
# 使用方法:
#   $ ./run.sh <env_name> <build_timestamp>
# 参数:
#   env_name:环境变量 build_timestamp:构建时间戳

set -eu

# 获取脚本的绝对路径（包括文件名）
script_path=$(readlink -f "$0")
# 获取脚本所在的目录的绝对路径
script_dir=$(dirname "$script_path")

# 函数定义
function print_usage {
  # 打印使用说明
  sed -n '2,7p' "$0"
}

function handle {
  local env_name=$1
  local build_timestamp=$2

  echo "执行的环境变量: ${env_name}"
  echo "执行的构建时间戳: ${build_timestamp}"
  if [ ! -f "$script_dir/env/env.${env_name}" ]; then
    echo "错误: 文件 'env.${env_name}' 不存在." >&2
    exit 1
  fi
  . $script_dir/env/env.${env_name}

  local dir="${root_dir}/${svc_name}/${svc_env}"
  local name="${svc_name}-${svc_env}"
  local version=${version}-${build_timestamp}
  local internal_ip="${internal_ip}"

  echo "---运行容器: ${name}:${version}---"
  # 构建 JSON 数据
  json_data="{ \
      \"container_name\": \"$name\", \
      \"image_version\": \"$domain/$namespace/$name:$version\", \
      \"ports\": [\"$svc_port:$svc_port\"], \
      \"mount_infos\": [\"/etc/localtime:/etc/localtime:ro\", \"$dir/logs:$svc_logs\"] \
  }"

  # 执行 curl 请求，并将响应结果存储在 response 变量中
  response=$(curl -s -X POST -H "Content-Type: application/json" -d "$json_data" "http://$internal_ip:10080/start-container")
  echo "响应结果: ${response}"
  # 检查响应中是否包含 "succeed"
  if echo "$response" | grep -q "successfully"; then
      echo "部署成功"
      exit 0
  else
      echo "部署失败"
      exit 1
  fi
}

# 主程序入口点
function main {
  if [ "$#" -ne 2 ]; then
    print_usage
    echo "错误: 需要提供两个个参数 <env_name> <build_timestamp>" >&2
    exit 1
  fi

  local env_name=$1
  local build_timestamp=$2

  handle "${env_name}" "${build_timestamp}"
}

# 错误处理
trap 'echo 发生了错误，脚本中断.' ERR

# 调用主函数
main "$@"
