package com.zzsn.leaderbase.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.copier.CopyOptions;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.zzsn.leaderbase.entity.CharacterBasicInfo;
import com.zzsn.leaderbase.entity.CharacterCategoryStructure;
import com.zzsn.leaderbase.mapper.CharacterBasicInfoMapper;
import com.zzsn.leaderbase.service.CharacterBasicInfoService;
import com.zzsn.leaderbase.service.CharacterTagService;
import com.zzsn.leaderbase.service.IGeneratorIdService;
import com.zzsn.leaderbase.service.LeaderCategoryService;
import com.zzsn.leaderbase.util.ExcelUtil;
import com.zzsn.leaderbase.vo.BasicInfoListVo;
import com.zzsn.leaderbase.vo.CharacterVo;
import com.zzsn.leaderbase.vo.Result;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/2/29 10:35
 * @Content: 人物基本信息
 */
@Slf4j
@Service
public class CharacterBasicInfoServiceImpl extends ServiceImpl<CharacterBasicInfoMapper, CharacterBasicInfo> implements CharacterBasicInfoService {

    //生成id
    @Autowired
    IGeneratorIdService generatorIdService;

    @Autowired
    CharacterTagService characterTagService;
    
    @Autowired
    LeaderCategoryService leaderCategoryService;

    @Override
    public CharacterVo getByUid(String uid) {
        return baseMapper.getByUid(uid);
    }

    @Override
    public CharacterBasicInfo getOneById(String id) {
        CharacterBasicInfo basicInfo = this.getById(id);
        return basicInfo;
    }

    @Override
    public IPage<CharacterBasicInfo> getList(BasicInfoListVo basicInfoListVo) {
        List<CharacterBasicInfo> list = baseMapper.getList(basicInfoListVo);
        Integer total = baseMapper.getListCount(basicInfoListVo);
        List<CharacterBasicInfo> listNew =setListValues(list,basicInfoListVo);
        IPage page = new Page();
        page.setRecords(listNew);
        page.setTotal(total);
        Integer pageNo = basicInfoListVo.getPageNo();
        Integer pageSize= basicInfoListVo.getPageSize();
        page.setCurrent(pageNo);
        page.setSize(pageSize);
        return page;
    }
    
    private List<CharacterBasicInfo> setListValues(List<CharacterBasicInfo> list, BasicInfoListVo basicInfoListVo) {
        //查询结果替换主数据
        List<CharacterBasicInfo> listNew=new ArrayList<>(list.size());
        for (CharacterBasicInfo characterBasicInfo : list) {
            Integer mainEntry = characterBasicInfo.getMainEntry();
            if(mainEntry == 0){
                QueryWrapper<CharacterBasicInfo> queryWrapper = new QueryWrapper<>();
                queryWrapper.eq("uid", characterBasicInfo.getUid())
                        .eq("main_entry",1);
                List<CharacterBasicInfo> mainList = this.list(queryWrapper);
                if(mainList.size() > 0){
                    listNew.add(mainList.get(0));
                } else {
                    listNew.add(characterBasicInfo);
                }
            } else {
                listNew.add(characterBasicInfo);
            }
        }
        //判断是否有关联数据
        hasChild(listNew);
        //记录热门标签
        String tagId = basicInfoListVo.getTagId();
        if(tagId != null && tagId.length() >0){
            String[] tagIdArr = tagId.split(",");
            for (String s : tagIdArr) {
                characterTagService.addTagTop(s);
            }
        }
        return list;
    }
    
    //判断是否有关联数据
    private List<CharacterBasicInfo> hasChild(List<CharacterBasicInfo> list){
        if(list != null && list.size() > 0 ){
            List<String> uidList = new ArrayList<>();
            list.forEach(info -> { uidList.add(info.getUid()); });
            QueryWrapper<CharacterBasicInfo> queryWrapper = new QueryWrapper<>();
            queryWrapper.in("uid", uidList)
                    .ne("main_entry",1).select("id","uid");
            List<CharacterBasicInfo> hasChildList = this.list(queryWrapper);
            if(hasChildList.size() > 0){
                Map<String, List<CharacterBasicInfo>> groupList = hasChildList.stream().collect(Collectors.groupingBy(e -> e.getUid()));
                list.forEach(info -> {
                    String uid = info.getUid();
                    List<CharacterBasicInfo> groupUid = groupList.get(uid);
                    if(groupUid != null && groupUid.size() != 0){
                        info.setHasChild(1);
                    }
                });
            }
        }
        return list;
    }


    @Override
    public List<CharacterVo> checkList(String articleId, String checkResult) {
        return baseMapper.checkList(articleId,checkResult);
    }

    @Override
    public List<CharacterBasicInfo> getListByName(String name) {
        return baseMapper.getListByName(name);
    }

    @Override
    public Integer deduplication(String duty, String company, String name) {
        Integer count = baseMapper.deduplication(duty, company, name);
        return count;
    }

    @Override
    public List<CharacterBasicInfo> getData(String time) {
        LambdaQueryWrapper<CharacterBasicInfo> lambdaQuery = Wrappers.lambdaQuery();
        if(!"".equals(time)){
            lambdaQuery.ge(CharacterBasicInfo::getCreateTime, time);
        }
        lambdaQuery.orderByAsc(CharacterBasicInfo::getCreateTime);
        lambdaQuery.select(CharacterBasicInfo::getUid,CharacterBasicInfo::getName);
        List<CharacterBasicInfo> list = baseMapper.selectList(lambdaQuery);
        return list;
    }

    @Override
    public List<CharacterBasicInfo> getPreviousDuty(String uid) {
        return baseMapper.getPreviousDuty(uid);
    }

    @Override
    public List<String> getCorrelationByUid(String uid) {
        return baseMapper.getCorrelationByUid(uid);
    }

    @Override
    public List<CharacterVo> getByOriginalId(String id) {
        return baseMapper.getByOriginalId(id);
    }

    @Override
    public Result<?> add(CharacterBasicInfo characterBasicInfo) {
        String id = Long.toString(generatorIdService.getOrderId());
        characterBasicInfo.setId(id);
        characterBasicInfo.setCreateTime(new Date());

        String uid = generatorIdService.getIdNo();
        characterBasicInfo.setUid(uid);
        characterBasicInfo.setMainEntry(1);

        Boolean res = this.save(characterBasicInfo);
        return Result.OK(res);
    }

    @Override
    public Result<?> edit(CharacterBasicInfo characterBasicInfo) {
        characterBasicInfo.setUpdateTime(new Date());
        Boolean res = this.updateById(characterBasicInfo);
        return Result.OK(res);
    }

    @Override
    public Boolean del(String id) {
        List<String> ids = Arrays.asList(id.split(","));
        return this.removeByIds(ids);
    }

    @Override
    public Result<?> correlation(CharacterBasicInfo characterBasicInfo) {
        //传过来的id是操作的数据id
        String id = characterBasicInfo.getId();
        CharacterBasicInfo newInfo = this.getById(id);

        String newId = Long.toString(generatorIdService.getOrderId());
        characterBasicInfo.setId(newId);
        characterBasicInfo.setMainEntry(0);
        characterBasicInfo.setCategory(newInfo.getCategory());
        characterBasicInfo.setCreateTime(new Date());
        Boolean res = this.save(characterBasicInfo);
        return Result.OK(res);
    }

    @Override
    public Result<?> correlationEdit(CharacterBasicInfo characterBasicInfo) {
        characterBasicInfo.setUpdateTime(new Date());
        Boolean res = this.updateById(characterBasicInfo);
        return Result.OK(res);
    }

    /**
     * 根据社会信用代码查询公司名称
     * @param socialCreditCode
     * @return
     */
    @Override
    public String getDepartmentCreditCode(String socialCreditCode) {
        return baseMapper.getDepartmentCreditCode(socialCreditCode);
    }

    /**
     * 根据社会信用代码查询相关联高管人员
     * @param socialCreditCode
     * @return
     */
    @Override
    public List<CharacterBasicInfo> getBasicInfoByCreditCode(String socialCreditCode) {
        return baseMapper.getBasicInfoByCreditCode(socialCreditCode);
    }

    @Override
    public List<Map<String, String>> getDepartment(String socialCreditCode, String department, String keyword) {
        if(keyword != null && keyword.length() > 0){
            if((keyword.length() == 18 || keyword.length() == 19) && keyword.matches("[0-9].*|[Zz].*")){
                socialCreditCode = keyword;
            } else {
                department = keyword;
            }
            keyword = null;
        }
        return baseMapper.getDepartment(socialCreditCode, department, keyword);
    }
    
    @Override
    public List<CharacterBasicInfo> getAllList(BasicInfoListVo basicInfoListVo) {
        List<CharacterBasicInfo> list = baseMapper.getAllList(basicInfoListVo);
        List<CharacterBasicInfo> listNew =setListValues(list,basicInfoListVo);
        return listNew;
    }

    @Override
    public CharacterBasicInfo getBasicInfoByUid(String category,String uid) {
        LambdaQueryWrapper<CharacterBasicInfo> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(CharacterBasicInfo::getUid,uid);
        queryWrapper.eq(CharacterBasicInfo::getCategory,category);
        return baseMapper.selectOne(queryWrapper);
    }

    @Override
    public CharacterBasicInfo getByNameAndByDepartment(String category, String typeId, String name, String department) {
        LambdaQueryWrapper<CharacterBasicInfo> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(CharacterBasicInfo::getTypeId,typeId);
        queryWrapper.eq(CharacterBasicInfo::getCategory,category);
        queryWrapper.eq(CharacterBasicInfo::getName,name);
        queryWrapper.eq(CharacterBasicInfo::getDepartment,department);
        List<CharacterBasicInfo> characterBasicInfos = baseMapper.selectList(queryWrapper);
        if(characterBasicInfos!=null && characterBasicInfos.size()>0){
            return characterBasicInfos.get(0);
        }
        return null;
    }
    
    @Override
    public String importExcel(String filePath, String category, Map<String, String> importFieldMap) {
        List<String> sheetNameList = new ArrayList<>();
        Workbook workbook = null;
        InputStream in = null;
        String result = "";
        try {
            // 获取输入流
            in = Files.newInputStream(Paths.get(filePath));
            // 判断excel版本
            if (ExcelUtil.judegExcelEdition(filePath)) {
                workbook = new XSSFWorkbook(in);
            } else {
                workbook = new HSSFWorkbook(in);
            }
            Iterator<Sheet> it = workbook.sheetIterator();
            while (it.hasNext()) {
                Sheet sheetTmp = it.next();
                String sheetName = sheetTmp.getSheetName();
                sheetNameList.add(sheetName);
            }
            Map<String, String> typeNameKeyMap = getTypeMapper(category).stream().collect(Collectors.toMap(CharacterCategoryStructure::getTypeName, CharacterCategoryStructure::getId));
            //人物类别映射集合，用来转化typeId和类别名称
            for (String sheetName : sheetNameList) {
                Sheet sheet = workbook.getSheet(sheetName);
                //获取标题行
                Row titleRow = sheet.getRow(0);
                List<CharacterBasicInfo> list = new ArrayList<>();
                for (int i = 1; i < sheet.getPhysicalNumberOfRows(); i++) {
                    // 循环获取工作表的每一行
                    Row sheetRow = sheet.getRow(i);
                    if (sheetRow == null) {
                        continue;
                    }
                    Map<String, Object> beanMap = new HashMap<>();
                    for (Cell cell : titleRow) {
                        String titleName = cell.getStringCellValue();
                        if (importFieldMap.containsKey(titleName)) {
                            try {
                                String stringCellValue = sheetRow.getCell(cell.getColumnIndex()).getStringCellValue();
                                //如果是分类列，转换成id
                                if ("分类".equals(titleName) && "typeId".equals(importFieldMap.get(titleName))) {
                                    stringCellValue = typeNameKeyMap.get(stringCellValue);
                                }
                                //将导入文件中不为null的列放入beanMap，key为导入列在数据库中对应的表列名
                                if (StringUtils.isNotEmpty(stringCellValue)) {
                                    beanMap.put(importFieldMap.get(titleName), stringCellValue);
                                }
                            } catch (Exception e) {
                                log.debug("数据导入excel解析{}行{}字段为空", cell.getColumnIndex(), titleName);
                            }
                        }
                    }
                    System.out.println(beanMap);
                    //过滤掉根据导入文档类型中文名找不到对应类别的数据（这些不做处理）
                    if (StringUtils.isNotEmpty((CharSequence) beanMap.get("typeId"))) {
                        list.add(BeanUtil.mapToBean(beanMap, CharacterBasicInfo.class, true, CopyOptions.create().setIgnoreError(true)));
                    }
                }
                //需要新增的数据列表
                List<CharacterBasicInfo> saveList = new ArrayList<>();
                //需要更新的数据列表
                List<CharacterBasicInfo> updateList = new ArrayList<>();
                for (CharacterBasicInfo basicInfo : list) {
                    if (basicInfo == null || StringUtils.isEmpty(basicInfo.getName()) || StringUtils.isEmpty(basicInfo.getDepartment()) || StringUtils.isEmpty(basicInfo.getTypeId())) {
                        log.info("数据为空，跳过");
                        continue;
                    }
                    basicInfo.setCategory(category);
                    if (StringUtils.isNotEmpty(basicInfo.getBirthday())) {
                        basicInfo.setBirthday(basicInfo.getBirthday().replaceAll("[./]", "-"));
                    }
                    if (StringUtils.isNotEmpty(basicInfo.getTechnicalDate())) {
                        basicInfo.setTechnicalDate(basicInfo.getTechnicalDate().replaceAll("[./]", "-"));
                    }
                    if (StringUtils.isNotEmpty(basicInfo.getWorkDate())) {
                        basicInfo.setWorkDate(basicInfo.getWorkDate().replaceAll("[./]", "-"));
                    }
                    //检查是否已有数据
                    CharacterBasicInfo oldBasicInfo = null;
                    if (StringUtils.isNotEmpty(basicInfo.getUid())) {
                        oldBasicInfo = getBasicInfoByUid(category, basicInfo.getUid());
                    } else {
                        oldBasicInfo = getByNameAndByDepartment(category, basicInfo.getTypeId(), basicInfo.getName(), basicInfo.getDepartment());
                    }
                    if (oldBasicInfo != null) {
                        //合并两个对象
                        oldBasicInfo.setSex(basicInfo.getSex());
                        oldBasicInfo.setBirthday(basicInfo.getBirthday());
                        oldBasicInfo.setNativePlace(basicInfo.getNativePlace());
                        oldBasicInfo.setSchool(basicInfo.getSchool());
                        oldBasicInfo.setSchoolSpeciality(basicInfo.getSchoolSpeciality());
                        oldBasicInfo.setEducation(basicInfo.getEducation());
                        oldBasicInfo.setDepart(basicInfo.getDepart());
                        oldBasicInfo.setDuty(basicInfo.getDuty());
                        oldBasicInfo.setTakeOfficeTime(basicInfo.getTakeOfficeTime());
                        oldBasicInfo.setTakeOfficeTimeEnd(basicInfo.getTakeOfficeTimeEnd());
                        oldBasicInfo.setExpertType(basicInfo.getExpertType());
                        oldBasicInfo.setResearchField(basicInfo.getResearchField());
                        oldBasicInfo.setSpeciality(basicInfo.getSpeciality());
                        oldBasicInfo.setTalentPlanning(basicInfo.getTalentPlanning());
                        oldBasicInfo.setTechnicalTitles(basicInfo.getTechnicalTitles());
                        oldBasicInfo.setTechnicalDate(basicInfo.getTechnicalDate());
                        oldBasicInfo.setWorkDate(basicInfo.getWorkDate());
                        oldBasicInfo.setUpdateTime(new Date());
                        updateList.add(oldBasicInfo);
                    } else {
                        String id = Long.toString(generatorIdService.getOrderId());
                        basicInfo.setId(id);
                        basicInfo.setCreateTime(new Date());
                        String uid = generatorIdService.getIdNo();
                        basicInfo.setUid(uid);
                        basicInfo.setMainEntry(1);
                        saveList.add(basicInfo);
                    }
                }
                if (!saveList.isEmpty()) {
                    saveBatch(saveList);
                    log.info("新增{}条数据", saveList.size());
                    result = "新增" + saveList.size() + "条数据";
                }
                if (!updateList.isEmpty()) {
                    updateBatchById(updateList);
                    log.info("更新{}条数据", updateList.size());
                    if (!"".equals(result)) {
                        result += ";";
                    }
                    result += "更新" + updateList.size() + "条数据";
                }
                if(saveList.isEmpty()&&updateList.isEmpty())
                {
                    result = "无可处理数据(如有数据代表数据不规范)";
                }
            }
        } catch (Exception e) {
            log.info("导入专家信息异常：{}", e.getMessage(), e);
            result = "导入专家信息异常：" + e.getMessage();
        } finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
            try {
                if (workbook != null) {
                    workbook.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return result;
    }
    
    private List<CharacterCategoryStructure> getTypeMapper(String category) {
        //根据名称模糊查询
        LambdaQueryWrapper<CharacterCategoryStructure> characterCategoryStructureLambdaQueryWrapper = Wrappers.lambdaQuery();
        characterCategoryStructureLambdaQueryWrapper.eq(StrUtil.isNotBlank(category), CharacterCategoryStructure::getCategory, category);
        List<CharacterCategoryStructure> characterCategoryWrapper = leaderCategoryService.list(characterCategoryStructureLambdaQueryWrapper);
        return characterCategoryWrapper;
    }
    
    @Override
    public ResponseEntity<byte[]> exportExcel(String category, List<CharacterBasicInfo> characterBasicInfos, Map<String, String> importFieldMap) throws IOException {
        return exportExcelByList(category, characterBasicInfos, importFieldMap);
    }
    
    @Override
    public ResponseEntity<byte[]> exportExcelByIds(String category, String ids, Map<String, String> importFieldMap) throws IOException {
        List<CharacterBasicInfo> characterBasicInfos = listByIds(Arrays.asList(ids.split(",")));
        return exportExcelByList(category, characterBasicInfos, importFieldMap);
    }
    
    @Override
    public ResponseEntity<byte[]> exportTemplate(Map<String, String> importFieldMap) throws IOException {
        return generateTemplate(importFieldMap);
    }
    
    private ResponseEntity<byte[]> generateTemplate(Map<String, String> importFieldMap) throws IOException {
        // 创建Excel工作簿
        Workbook workbook = new XSSFWorkbook();
        Sheet sheet = workbook.createSheet("人物信息");
        // 填充数据到Excel工作表
        Row titlerow = sheet.createRow(0);
        // 创建字体样式
        Font headerFont = workbook.createFont();
        headerFont.setFontName("Arial");
        headerFont.setFontHeightInPoints((short) 12);
        headerFont.setBold(true);
        // 创建单元格样式并应用字体
        CellStyle titleCellStyle = workbook.createCellStyle();
        titleCellStyle.setFont(headerFont);
        titleCellStyle.setAlignment(HorizontalAlignment.CENTER);
        titleCellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        titleCellStyle.setFillForegroundColor(IndexedColors.BRIGHT_GREEN.getIndex());
        titleCellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        AtomicInteger colNum = new AtomicInteger();
        importFieldMap.forEach((key, value) -> {
            int andIncrement = colNum.getAndIncrement();
            Cell cell = titlerow.createCell(andIncrement);
            cell.setCellValue(key);
            cell.setCellStyle(titleCellStyle);
            sheet.setColumnWidth(andIncrement, 15 * 256);
        });
        // 将Excel工作簿写入字节数组输出流
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        workbook.write(byteArrayOutputStream);
        // 准备HTTP响应头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.parseMediaType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"));
        // 对文件名进行编码
        String encodedFileName = URLEncoder.encode("人物导入模板.xlsx", "utf-8");
        headers.add("Content-Disposition", "attachment; filename=" + encodedFileName);
        // 返回Excel文件作为字节数组响应
        return new ResponseEntity<>(byteArrayOutputStream.toByteArray(), headers, HttpStatus.OK);
    }
    
    private ResponseEntity<byte[]> exportExcelByList(String category,List<CharacterBasicInfo> characterBasicInfos, Map<String, String> importFieldMap) throws IOException {
        // 创建Excel工作簿
        Workbook workbook = new XSSFWorkbook();
        Sheet sheet = workbook.createSheet("人物信息");
        // 填充数据到Excel工作表
        Row titlerow = sheet.createRow(0);
        // 创建字体样式
        Font headerFont = workbook.createFont();
        headerFont.setFontName("Arial");
        headerFont.setFontHeightInPoints((short) 12);
        headerFont.setBold(true);
        // 创建单元格样式并应用字体
        CellStyle titleCellStyle = workbook.createCellStyle();
        titleCellStyle.setFont(headerFont);
        titleCellStyle.setAlignment(HorizontalAlignment.CENTER);
        titleCellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        titleCellStyle.setFillForegroundColor(IndexedColors.BRIGHT_GREEN.getIndex());
        titleCellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        AtomicInteger colNum = new AtomicInteger();
        importFieldMap.forEach((key, value) -> {
            int andIncrement = colNum.getAndIncrement();
            Cell cell = titlerow.createCell(andIncrement);
            cell.setCellValue(key);
            cell.setCellStyle(titleCellStyle);
            sheet.setColumnWidth(andIncrement, 15 * 256);
        });
        int rowNum = 1;
        Map<String,String> typeIdeKeyMap=getTypeMapper(category).stream().collect(Collectors.toMap(CharacterCategoryStructure::getId, CharacterCategoryStructure::getTypeName));
        //数据填充到Excel工作表
        for (CharacterBasicInfo characterBasicInfo : characterBasicInfos) {
            Row row = sheet.createRow(rowNum++);
            //总列数
            int columns = importFieldMap.size();
            for (int i = 0; i < columns; i++) {
                Cell cell = row.createCell(i);
                // 列名获取
                String columnName = importFieldMap.get(titlerow.getCell(i).getStringCellValue());
                // 对象转化为map
                Map<String, Object> characterMap = BeanUtil.beanToMap(characterBasicInfo);
                if (columnName != null) {
                    Object value = characterMap.get(columnName);
                    if("typeId".equals(columnName))
                    {
                        value=typeIdeKeyMap.get(value);
                    }
                    if (value != null) {
                        if (value instanceof String) {
                            cell.setCellValue((String) value);
                        } else if (value instanceof Number) {
                            cell.setCellValue(((Number) value).doubleValue());
                        } else if (value instanceof Date) {
                            cell.setCellValue((Date) value);
                        } else {
                            // 处理其他类型的值，例如布尔值等
                            cell.setCellValue(value.toString());
                        }
                    } else {
                        // 如果值为 null，可以设置为空字符串或其他默认值
                        cell.setCellValue("");
                    }
                } else {
                    // 如果 columnName 不存在于 importFieldMap 中，可以设置为空字符串或其他默认值
                    cell.setCellValue("");
                }
            }
        }
        // 将Excel工作簿写入字节数组输出流
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        workbook.write(byteArrayOutputStream);
        // 准备HTTP响应头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.parseMediaType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"));
        // 对文件名进行编码
        String encodedFileName = URLEncoder.encode("人物信息.xlsx", "utf-8");
        headers.add("Content-Disposition", "attachment; filename=" + encodedFileName);
        // 返回Excel文件作为字节数组响应
        return new ResponseEntity<>(byteArrayOutputStream.toByteArray(), headers, HttpStatus.OK);
    }
}
