package com.zzsn.leaderbase.controller;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.copier.CopyOptions;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.zzsn.leaderbase.entity.CharacterBasicInfo;
import com.zzsn.leaderbase.service.CharacterBasicInfoService;
import com.zzsn.leaderbase.service.IGeneratorIdService;
import com.zzsn.leaderbase.util.ExcelUtil;
import com.zzsn.leaderbase.util.SortUtil;
import com.zzsn.leaderbase.vo.BasicInfoListVo;
import com.zzsn.leaderbase.vo.Result;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;

@Slf4j
@RestController
@RequestMapping("/api/character/basic")
public class CharacterBasicInfoController {
    
    @Autowired
    CharacterBasicInfoService characterBasicInfoService;
    @Autowired
    IGeneratorIdService generatorIdService;

    private static Map<String, String> importFieldMap;
    /**
     * 资讯抽取人物信息分页查询
     * 人物审核列表
     *
     * @param
     * @return
     */
    @PostMapping("/list")
    public Result<?> getList(@RequestBody BasicInfoListVo basicInfoListVo) {
        log.info("人物管理列表:{}", basicInfoListVo);
        Integer office = basicInfoListVo.getPageSize() * (basicInfoListVo.getPageNo() - 1);
        basicInfoListVo.setOffset(office);
        basicInfoListVo.setColumn(SortUtil.humpToLine(basicInfoListVo.getColumn()));
        IPage page = characterBasicInfoService.getList(basicInfoListVo);
        return Result.OK(page);
    }
    
    @GetMapping("/getById")
    public Result<?> getById(@RequestParam(name = "id", required = true) String id) {
        log.info("人物审核根据人物id查询:{}", id);
        CharacterBasicInfo info = characterBasicInfoService.getOneById(id);
        return Result.OK(info);
    }
    
    /**
     * 根据姓名查询 疑似人物查询
     *
     * @param name
     * @return
     */
    @GetMapping("/getListByName")
    public Result<?> getListByName(@RequestParam(name = "name", required = true) String name) {
        log.info("根据人物名称查询疑似人物:{}", name);
        List<CharacterBasicInfo> list = characterBasicInfoService.getListByName(name);
        return Result.OK(list);
    }
    
    /**
     * 新增人物
     */
    @PostMapping("/add")
    public Result<?> add(@RequestBody CharacterBasicInfo characterBasicInfo) {
        log.info("新增人物:{}", characterBasicInfo);
        return characterBasicInfoService.add(characterBasicInfo);
    }
    
    /**
     * 修改人物
     */
    @PostMapping("/edit")
    public Result<?> edit(@RequestBody CharacterBasicInfo characterBasicInfo) {
        log.info("新增人物:{}", characterBasicInfo);
        String id = characterBasicInfo.getId();
        if (id == null || id.length() == 0)
            return Result.error("人物id不能为空");
        return characterBasicInfoService.edit(characterBasicInfo);
    }
    
    /**
     * 根据id删除人物
     *
     * @param id
     * @return
     */
    @GetMapping("/del")
    public Result<?> del(@RequestParam(name = "id", required = true) String id) {
        log.info("根据id删除人物:{}", id);
        Boolean res = characterBasicInfoService.del(id);
        return Result.OK(res);
    }
    
    /**
     * 根据uid查询人物的历任职务
     *
     * @param uid
     * @return
     */
    @GetMapping("/getPreviousDuty")
    public Result<?> getPreviousDuty(@RequestParam(name = "uid", required = true) String uid) {
        log.info("根据uid查询人物的历任职务:{}", uid);
        List<CharacterBasicInfo> list = characterBasicInfoService.getPreviousDuty(uid);
        return Result.OK(list);
    }
    
    /**
     * 关联
     *
     * @param characterBasicInfo
     * @return
     */
    @PostMapping("/correlation")
    public Result<?> correlation(@RequestBody CharacterBasicInfo characterBasicInfo) {
        log.info("关联:{}", characterBasicInfo.getId());
        return characterBasicInfoService.correlation(characterBasicInfo);
    }
    
    /**
     * 修改关联人物
     *
     * @param characterBasicInfo
     * @return
     */
    @PostMapping("/correlationEdit")
    public Result<?> correlationEdit(@RequestBody CharacterBasicInfo characterBasicInfo) {
        log.info("修改关联人物:{}", characterBasicInfo.getId());
        return characterBasicInfoService.correlationEdit(characterBasicInfo);
    }
    
    /**
     * 企业查询
     *
     * @param socialCreditCode 社会信用代码
     * @param department
     * @return
     */
    @GetMapping("/getDepartment")
    public Result<?> getDepartment(@RequestParam(name = "socialCreditCode", required = false) String socialCreditCode,
                                   @RequestParam(name = "department", required = false) String department,
                                   @RequestParam(name = "keyword", required = false) String keyword) {
        log.info("查询企业名称:{}---{}---{}", socialCreditCode, department, keyword);
        List<Map<String, String>> list = characterBasicInfoService.getDepartment(socialCreditCode, department, keyword);
        return Result.OK(list);
    }
    
    /**
     * @param basicInfoListVo 条件封装
     * @描述 查询专家列表
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/listAll")
    public Result<?> listAll(@RequestBody BasicInfoListVo basicInfoListVo) {
        //类别过滤为专家
        basicInfoListVo.setCategory("2");
        basicInfoListVo.setColumn(SortUtil.humpToLine(basicInfoListVo.getColumn()));
        List<CharacterBasicInfo> list= characterBasicInfoService.getAllList(basicInfoListVo);
        return Result.OK(list);
    }
    
    /**
     * @param basicInfoListVo 条件封装
     * @描述 查询专家列表（分页，产业链关联专家时查询用）
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/pageListAll")
    public Result<?> pageListAll(@RequestBody BasicInfoListVo basicInfoListVo) {
        //类别过滤为专家
        basicInfoListVo.setCategory("2");
        Integer office = basicInfoListVo.getPageSize() * (basicInfoListVo.getPageNo() - 1);
        basicInfoListVo.setOffset(office);
        basicInfoListVo.setColumn(SortUtil.humpToLine(basicInfoListVo.getColumn()));
        IPage page = characterBasicInfoService.getList(basicInfoListVo);
        return Result.OK(page);
    }
    @GetMapping("/importExcel")
    public void importExcel(String[] args) {
        //本地文件路径
        String filePath = "E:\\数能软件\\克虏宝\\人物服务\\专家导入数据2025-01-08.xlsx";
        addSummaryLocalExcel(filePath,"2");
    }

    public void addSummaryLocalExcel(String filePath, String category) {
        List<String> sheetNameList = new ArrayList<>();
        Workbook workbook = null;
        InputStream in = null;
        try {
            // 获取输入流
            in = Files.newInputStream(Paths.get(filePath));
            // 判断excel版本
            if (ExcelUtil.judegExcelEdition(filePath)) {
                workbook = new XSSFWorkbook(in);
            } else {
                workbook = new HSSFWorkbook(in);
            }
            Iterator<Sheet> it = workbook.sheetIterator();
            while (it.hasNext()) {
                Sheet sheetTmp = it.next();
                String sheetName = sheetTmp.getSheetName();
                sheetNameList.add(sheetName);
            }
            for (String sheetName : sheetNameList) {
                Sheet sheet = workbook.getSheet(sheetName);
                //获取标题行
                Row titleRow = sheet.getRow(0);
                List<CharacterBasicInfo> list = new ArrayList<>();
                for (int i = 1; i < sheet.getPhysicalNumberOfRows(); i++) {
                    // 循环获取工作表的每一行
                    Row sheetRow = sheet.getRow(i);
                    if (sheetRow == null) {
                        continue;
                    }
                    Map<String, Object> beanMap = new HashMap<>();
                    for (Cell cell : titleRow) {
                        String titleName = cell.getStringCellValue();
                        if(importFieldMap.containsKey(titleName)){
                            try {
                                String stringCellValue = sheetRow.getCell(cell.getColumnIndex()).getStringCellValue();
                                if(StringUtils.isNotEmpty(stringCellValue)){
                                    beanMap.put(importFieldMap.get(titleName), stringCellValue);
                                }
                            }catch (Exception e){
                                log.debug("数据导入excel解析{}行{}字段为空",cell.getColumnIndex(),titleName);
                            }
                        }
                    }
                    System.out.println(beanMap);
                    list.add(BeanUtil.mapToBean(beanMap, CharacterBasicInfo.class, true,CopyOptions.create().setIgnoreError(true)));
                }
                //需要新增的数据列表
                List<CharacterBasicInfo> saveList = new ArrayList<>();
                //需要更新的数据列表
                List<CharacterBasicInfo> updateList = new ArrayList<>();
                for (CharacterBasicInfo basicInfo : list) {
                    if(basicInfo == null || StringUtils.isEmpty(basicInfo.getName()) || StringUtils.isEmpty(basicInfo.getDepartment())){
                        log.info("数据为空，跳过");
                        continue;
                    }
                    basicInfo.setCategory(category);
                    if(StringUtils.isNotEmpty(basicInfo.getBirthday())){
                        basicInfo.setBirthday(basicInfo.getBirthday().replaceAll("[./]","-"));
                    }
                    if(StringUtils.isNotEmpty(basicInfo.getTechnicalDate())){
                        basicInfo.setTechnicalDate(basicInfo.getTechnicalDate().replaceAll("[./]","-"));
                    }
                    if(StringUtils.isNotEmpty(basicInfo.getWorkDate())){
                        basicInfo.setWorkDate(basicInfo.getWorkDate().replaceAll("[./]","-"));
                    }
                    //检查是否已有数据
                    CharacterBasicInfo oldBasicInfo = null;
                    if(StringUtils.isNotEmpty(basicInfo.getUid())){
                        oldBasicInfo = characterBasicInfoService.getBasicInfoByUid(basicInfo.getUid());
                    }else{
                        oldBasicInfo = characterBasicInfoService.getByNameAndByDepartment(category, basicInfo.getName(), basicInfo.getDepartment());
                    }
                    if(oldBasicInfo != null){
                        //合并两个对象
                        oldBasicInfo.setSex(basicInfo.getSex());
                        oldBasicInfo.setBirthday(basicInfo.getBirthday());
                        oldBasicInfo.setNativePlace(basicInfo.getNativePlace());
                        oldBasicInfo.setSchool(basicInfo.getSchool());
                        oldBasicInfo.setSchoolSpeciality(basicInfo.getSchoolSpeciality());
                        oldBasicInfo.setEducation(basicInfo.getEducation());
                        oldBasicInfo.setDepart(basicInfo.getDepart());
                        oldBasicInfo.setDuty(basicInfo.getDuty());
                        oldBasicInfo.setTakeOfficeTime(basicInfo.getTakeOfficeTime());
                        oldBasicInfo.setTakeOfficeTimeEnd(basicInfo.getTakeOfficeTimeEnd());
                        oldBasicInfo.setExpertType(basicInfo.getExpertType());
                        oldBasicInfo.setResearchField(basicInfo.getResearchField());
                        oldBasicInfo.setSpeciality(basicInfo.getSpeciality());
                        oldBasicInfo.setTalentPlanning(basicInfo.getTalentPlanning());
                        oldBasicInfo.setTechnicalTitles(basicInfo.getTechnicalTitles());
                        oldBasicInfo.setTechnicalDate(basicInfo.getTechnicalDate());
                        oldBasicInfo.setWorkDate(basicInfo.getWorkDate());
                        oldBasicInfo.setUpdateTime(new Date());
                        updateList.add(oldBasicInfo);
                    }else {
                        String id = Long.toString(generatorIdService.getOrderId());
                        basicInfo.setId(id);
                        basicInfo.setCreateTime(new Date());
                        String uid = generatorIdService.getIdNo();
                        basicInfo.setUid(uid);
                        basicInfo.setMainEntry(1);
                        saveList.add(basicInfo);
                    }
                }
                if(!saveList.isEmpty()){
                    characterBasicInfoService.saveBatch(saveList);
                    log.info("新增{}条数据",saveList.size());
                }
                if(!updateList.isEmpty()){
                    characterBasicInfoService.updateBatchById(updateList);
                    log.info("更新{}条数据",updateList.size());
                }
            }
        } catch (Exception e) {
            log.info("导入专家信息异常：{}",e.getMessage(), e);
        } finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
            try {
                if (workbook != null) {
                    workbook.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }
    static {
        importFieldMap = new HashMap<>();
        importFieldMap.put("人物id", "uid");
        importFieldMap.put("姓名", "name");
        importFieldMap.put("性别", "sex");
        importFieldMap.put("出生年月", "birthday");
        importFieldMap.put("籍贯", "nativePlace");
        importFieldMap.put("毕业院校", "school");
        importFieldMap.put("毕业专业", "schoolSpeciality");
        importFieldMap.put("学历", "education");
        importFieldMap.put("单位名称", "department");
        importFieldMap.put("所在部门", "depart");
        importFieldMap.put("职务", "duty");
        importFieldMap.put("任职开始时间", "takeOfficeTime");
        importFieldMap.put("任职结束时间", "takeOfficeTimeEnd");
        importFieldMap.put("顶尖专家/领军专家", "expertType");
        importFieldMap.put("研究领域", "researchField");
        importFieldMap.put("专业方向", "speciality");
        importFieldMap.put("已入选的人才计划", "talentPlanning");
        importFieldMap.put("专业技术职称", "technicalTitles");
        importFieldMap.put("专业技术职称取得时间", "technicalDate");
        importFieldMap.put("参加工作时间", "workDate");
    }
}

