package com.zzsn.knowbase.service.impl;

import com.zzsn.knowbase.entity.KnowFile;
import com.zzsn.knowbase.service.ILocalFileService;
import com.zzsn.knowbase.util.file.FileUtility;
import com.zzsn.knowbase.vo.Result;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDate;
import java.util.UUID;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/1/9 9:36
 * @Content:
 */
@Service
@Slf4j
public class LocalFileServiceImpl implements ILocalFileService {

    @Autowired
    private FileUtility fileUtility;

    @Value("${files.storage}")
    String filesStorage;

    @Override
    public Result<KnowFile> upload(MultipartFile file, String uid) {
        try {
            String fullFileName = file.getOriginalFilename();  // get file name
            String fileExtension = fileUtility.getFileExtension(fullFileName);  // get file extension
            long fileSize = file.getSize();  // get file size

            // check if the file size exceeds the maximum file size or is less than 0
            if (fileUtility.getMaxFileSize() < fileSize || fileSize <= 0) {
                Result result = Result.error("文件大小不正确！");
                return result;
            }

            // check if file extension is supported by the editor
            if (!fileUtility.getFileExts().contains(fileExtension)) {
                Result result = Result.error("不支持的文件类型！");
                return result;
            }

            String fileName = file.getOriginalFilename();
            String fileSuffix = getFileSuffix(fileName);

            uid = UUID.randomUUID().toString();

            String filePath = getFilePath() + uid + "." + fileSuffix;
            byte[] bytes = file.getBytes();  // get file in bytes
            //Files.write(Paths.get(filePath), bytes);
            file.transferTo(new File(filePath));

            KnowFile knowFile = new KnowFile();
            knowFile.setFileId(uid);
            knowFile.setFileName(fileName);
            knowFile.setFilePath(filePath);
            knowFile.setFileType(fileSuffix);
            knowFile.setFileSize(fileSize);
            Result result = Result.OK(knowFile);
            return result;  // create user metadata and return it
        } catch (Exception e) {
            e.printStackTrace();
        }
        // if the operation of file uploading is unsuccessful, an error occurs
        Result result = Result.error("上传文件时出现问题！");
        return result;
    }

    private String getFilePath(){
        LocalDate currentDate = LocalDate.now();
        //System.out.println("当前日期: " + currentDate);
        String filePath = filesStorage + currentDate + "/";

        //判断文件夹是否存在，不存在创建
        Path directory = Paths.get(filePath);
        if (!Files.exists(directory)) {
            try {
                Files.createDirectories(directory);
                log.info("文件夹创建成功：" + filePath);
            } catch (IOException e) {
                log.error("文件夹创建失败：" + filePath);
                e.printStackTrace();
            }
        }
        return filePath;
    }

    private String getFileSuffix(String fileName){
        int lastIndexOfDot = fileName.lastIndexOf('.');
        String fileExtension = "";

        if (lastIndexOfDot != -1) {
            fileExtension = fileName.substring(lastIndexOfDot + 1);
        }
        return fileExtension;
    }

}
