package com.zzsn.knowbase.service.impl;

import com.zzsn.knowbase.entity.KnowFile;
import com.zzsn.knowbase.service.ILocalFileService;
import com.zzsn.knowbase.util.file.FileUtility;
import com.zzsn.knowbase.vo.Result;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.core.io.UrlResource;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/1/9 9:36
 * @Content:
 */
@Service
@Slf4j
public class LocalFileServiceImpl implements ILocalFileService {

    @Autowired
    private FileUtility fileUtility;

    @Value("${files.storage}")
    String filesStorage;

    @Override
    public Result<KnowFile> upload(MultipartFile file) {
        try {
            String fullFileName = file.getOriginalFilename();  // get file name
            String fileExtension = fileUtility.getFileExtension(fullFileName);  // get file extension
            long fileSize = file.getSize();  // get file size

            // check if the file size exceeds the maximum file size or is less than 0
            if (fileUtility.getMaxFileSize() < fileSize || fileSize <= 0) {
                Result result = Result.error("文件大小不正确！");
                return result;
            }
            // check if file extension is supported by the editor
            if (!fileUtility.getFileExts().contains(fileExtension)) {
                Result result = Result.error("不支持的文件类型！");
                return result;
            }

            String fileName = file.getOriginalFilename();
            String fileSuffix = getFileSuffix(fileName);

            String uid = UUID.randomUUID().toString();

            String filePath = getFilePath() + uid + "." + fileSuffix;
            byte[] bytes = file.getBytes();  // get file in bytes
            //Files.write(Paths.get(filePath), bytes);
            file.transferTo(new File(filePath));

            KnowFile knowFile = new KnowFile();
            knowFile.setFileId(uid);
            knowFile.setFileName(fileName);
            knowFile.setFilePath(filePath);
            knowFile.setFileType(fileSuffix);
            knowFile.setFileSize(fileSize);
            Result result = Result.OK(knowFile);
            return result;  // create user metadata and return it
        } catch (Exception e) {
            e.printStackTrace();
        }
        // if the operation of file uploading is unsuccessful, an error occurs
        Result result = Result.error("上传文件时出现问题！");
        return result;
    }

    @Override
    public List<Result<KnowFile>> upload(Map<String, MultipartFile> fileMap) {
        List<Result<KnowFile>> list = new ArrayList<>();
        for (Map.Entry<String,MultipartFile> entity : fileMap.entrySet()) {
            MultipartFile file = entity.getValue();// 获取上传文件对象
            Result<KnowFile> result = upload(file);
            list.add(result);
        }
        return list;
    }



    /**
     * 文件下载
     */
    @Override
    public ResponseEntity<Resource> download(String fileName, String filePath) {
        Path path = Paths.get(filePath);
        try {
            Resource resource = new UrlResource(path.toUri());
            if (resource.exists() || resource.isReadable()) {
                return ResponseEntity.ok()
                        .header("Content-Disposition", "attachment; filename=\"" + fileName + "\"")
                        .body(resource);
            } else {
                throw new RuntimeException("文件不存在或不可读");
            }
        } catch (MalformedURLException e) {
            e.printStackTrace();
            throw new RuntimeException("文件读取失败");
        }
    }

    //获取文件夹路径
    private String getFilePath(){
        LocalDate currentDate = LocalDate.now();
        //System.out.println("当前日期: " + currentDate);
        String filePath = filesStorage + currentDate + "/";

        //判断文件夹是否存在，不存在创建
        Path directory = Paths.get(filePath);
        if (!Files.exists(directory)) {
            try {
                Files.createDirectories(directory);
                log.info("文件夹创建成功：" + filePath);
            } catch (IOException e) {
                log.error("文件夹创建失败：" + filePath);
                e.printStackTrace();
            }
        }
        return filePath;
    }
    //获取文件后缀
    private String getFileSuffix(String fileName){
        int lastIndexOfDot = fileName.lastIndexOf('.');
        String fileExtension = "";

        if (lastIndexOfDot != -1) {
            fileExtension = fileName.substring(lastIndexOfDot + 1);
        }
        return fileExtension;
    }
}
