package com.zzsn.event.controller;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.*;
import com.zzsn.event.enums.AnalysisColumnEnum;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.*;
import com.zzsn.event.util.CalculateUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.CoOccurrenceVO;
import com.zzsn.event.vo.CountVO;
import com.zzsn.event.vo.EventViewVO;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * 事件分析
 *
 * @author lkg
 * @date 2024/1/24
 */
@RestController
@RequestMapping("/analysis")
public class EventAnalysisController {

    @Autowired
    private EsStatisticsService esStatisticsService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private AnalysisService analysisService;
    @Autowired
    private EventAnalysisReportService eventAnalysisReportService;
    @Autowired
    private EsService esService;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private EventCollectMapService eventCollectMapService;
    @Autowired
    private EventContextService eventContextService;
    @Autowired
    private EventAnalysisVersionService eventAnalysisVersionService;
    @Autowired
    private EventAnalysisVersionRecordService eventAnalysisVersionRecordService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private IEventService eventService;


    /**
     * 重新生成
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/7/17
     */
    @GetMapping("/regenerate")
    public Result<?> regenerate(@RequestParam String eventId) {
        CompletableFuture.runAsync(() -> analysisService.regenerate(eventId));
        return Result.OK("重新生成中,请稍后查看", null);
    }

    /**
     * 事件分析的版本列表
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/7/16
     */
    @GetMapping("/versionList")
    public Result<?> versionList(@RequestParam String eventId) {
        LambdaQueryWrapper<EventAnalysisVersion> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(EventAnalysisVersion::getEventId, eventId);
        List<EventAnalysisVersion> versionList = eventAnalysisVersionService.list(queryWrapper);
        return Result.OK(versionList);
    }


    /**
     * 3.4 总体分析
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/24
     */
    @GetMapping("/total")
    public Result<?> totalAndMax(@RequestParam String eventId,
                                 @RequestParam(required = false) String startTime,
                                 @RequestParam(required = false) String endTime,
                                 @RequestParam(defaultValue = "1") Integer type) {
        Event event = eventService.getById(eventId);
        if (StringUtils.isEmpty(startTime)) {
            startTime = DateUtil.formatDateTime(event.getStartTime());
        }
        Map<String, String> map = esStatisticsService.totalAndMax(eventId, null, null, type);
        if (StringUtils.isEmpty(endTime)) {
            endTime = DateUtil.now();
        }
        long hours = DateUtil.between(DateUtil.parseDateTime(startTime), DateUtil.parseDateTime(endTime), DateUnit.HOUR);
        Object count = map.get("totalCount");
        String divide = CalculateUtil.divide(String.valueOf(count), String.valueOf(hours), 0, RoundingMode.UP);
        List<Map<String, Object>> dataList = new ArrayList<>();
        Map<String, Object> totalCountMap = new HashMap<>();
        totalCountMap.put("name", "事件资讯数量");
        totalCountMap.put("value", count);
        totalCountMap.put("unit", "条");
        dataList.add(totalCountMap);
        Map<String, Object> maxCountMap = new HashMap<>();
        maxCountMap.put("name", "波峰值");
        maxCountMap.put("value", map.get("max"));
        maxCountMap.put("unit", "条");
        dataList.add(maxCountMap);
        Map<String, Object> spreadMap = new HashMap<>();
        spreadMap.put("name", "平均传播速度");
        spreadMap.put("value", divide);
        spreadMap.put("unit", type == 1 ? "条/小时" : "条/天");
        dataList.add(spreadMap);
        return Result.OK(dataList);
    }

    /**
     * 事件脉络
     *
     * @param eventId   事件id
     * @param versionId 版本id
     * @author lkg
     * @date 2025/7/14
     */
    @GetMapping("/eventContext")
    public Result<?> eventContext(@RequestParam String eventId,
                                  @RequestParam(required = false) String versionId) {
        //if (StringUtils.isNotEmpty(versionId)) {
        //    String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.EVENT_CONTEXT.getCode());
        //    return Result.OK(JSON.parseArray(versionData,JSONObject.class));
        //}
        List<Map<String, Object>> versionData = eventContextService.eventContext(eventId);
        return Result.OK(versionData);
    }

    /**
     * 事件脉络关键节点列表
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/7/15
     */
    @GetMapping("/eventContext/keNodeList")
    public Result<?> keNodeList(@RequestParam String eventId) {
        List<EventContext> keNodeList = eventContextService.keNodeList(eventId);
        return Result.OK(keNodeList);
    }


    /**
     * 关键节点下的资讯列表
     *
     * @param eventId   事件id
     * @param keyNodeId 关键节点id
     * @param pageNo    页码
     * @param pageSize  每页返回数量
     * @author lkg
     * @date 2025/7/15
     */
    @GetMapping("/eventContext/contextArticleList")
    public Result<?> contextArticleList(@RequestParam String eventId,
                                        @RequestParam String keyNodeId,
                                        @RequestParam(defaultValue = "1") Integer pageNo,
                                        @RequestParam(defaultValue = "10") Integer pageSize) {
        Page<EventContext> page = eventContextService.contextArticleList(eventId, keyNodeId, pageNo, pageSize);
        return Result.OK(page);
    }


    /**
     * 3.8 媒体观点/专家观点信息列表
     *
     * @author lkg
     * @date 2024/1/12
     */
    @ApiOperation(value = "媒体观点/专家观点信息列表", notes = "媒体观点/专家观点信息列表")
    @GetMapping(value = "/listEventAnalysis")
    public Result<?> listEventAnalysis(@RequestParam(name = "eventId") String eventId,
                                       @RequestParam(name = "relationId") String relationId,
                                       @RequestParam(defaultValue = "1") Integer pageNo,
                                       @RequestParam(defaultValue = "10") Integer pageSize) {
        List<EventViewVO> viewpointList = esService.viewpointList(eventId, relationId, pageNo, pageSize);
        if (CollectionUtils.isNotEmpty(viewpointList)) {
            viewpointList.forEach(e -> e.setOrigin(Constants.getRealOrigin(e.getOrigin())));
        }
        return Result.OK(viewpointList);
    }

    /**
     * 热词分析
     *
     * @param eventId  事件id
     * @param pageSize 返回个数
     * @author lkg
     * @date 2025/7/8
     */
    @ApiOperation(value = "热词查询", notes = "热词查询")
    @GetMapping(value = "/keywordsCount")
    public Result<?> keywordsCount(@RequestParam String eventId,
                                   @RequestParam(required = false) String startTime,
                                   @RequestParam(required = false) String endTime,
                                   @RequestParam(defaultValue = "150") Integer pageSize) {
        List<CountVO> countVOS = analysisService.wordCloud(eventId, startTime, endTime, pageSize);
        return Result.OK(countVOS);
    }

    /**
     * 热词趋势分析
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/7/8
     */
    @GetMapping("/keywordTrend")
    public Result<?> keywordTrend(@RequestParam String eventId,
                                  @RequestParam(required = false) String startTime,
                                  @RequestParam(required = false) String endTime,
                                  @RequestParam(required = false) String versionId) {
        if (StringUtils.isNotEmpty(versionId)) {
            String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.KEYWORD_TREND.getCode());
            return Result.OK(JSON.parseArray(versionData,JSONObject.class));
        }
        List<CountVO> versionData = analysisService.wordTrend(eventId, startTime, endTime);
        return Result.OK(versionData);
    }


    /**
     * 热词共现
     *
     * @param eventId 事件id
     * @param startTime 开始时间
     * @param endTime 结束时间
     * @param versionId 版本id
     * @author lkg
     * @date 2025/7/9
     */
    @GetMapping("/coOccurrence")
    public Result<?> coOccurrence(@RequestParam String eventId,
                                  @RequestParam(required = false) String startTime,
                                  @RequestParam(required = false) String endTime,
                                  @RequestParam(required = false) String versionId) {
        if (StringUtils.isNotEmpty(versionId)) {
            String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.CO_OCCURRENCE.getCode());
            return Result.OK(JSON.parseArray(versionData,JSONObject.class));
        }
        //增加缓存逻辑，提升查询效率
        String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.CO_OCCURRENCE + eventId;
        Object cacheObject = redisUtil.get(key);
        if (cacheObject != null) {
            return Result.OK(cacheObject);
        } else {
            List<CoOccurrenceVO> coOccurrenceVOS = analysisService.coOccurrence(eventId, startTime, endTime);
            if (CollectionUtils.isNotEmpty(coOccurrenceVOS)) {
                redisUtil.set(key, coOccurrenceVOS,3600 * 24);
            }
            return Result.OK(coOccurrenceVOS);
        }
    }

    /**
     * 事件下情感类标签下资讯数量统计
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/5/9
     */
    @GetMapping("/orientationCount")
    public Result<?> orientationCount(@RequestParam String eventId) {
        List<CountVO> countVOS = esStatisticsService.orientationCount(eventId, null, null);
        return Result.OK(countVOS);
    }


    /**
     * 3.5 情感判断分析
     *
     * @param eventId   专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/orientation")
    public Result<?> orientation(@RequestParam String eventId, @RequestParam String startTime,
                                 @RequestParam(required = false) String endTime, @RequestParam Integer type) {
        List<CountVO> list = new ArrayList<>();
        Map<String, String> map = esStatisticsService.totalAndMax(eventId, null, null, type);
        String totalCount = map.get("totalCount");
        if (!totalCount.equals("0")) {
            String maxTime = map.get("maxTime");
            Map<String, String> timeRangeMap = getTimeRange(startTime, endTime, maxTime, type);
            startTime = timeRangeMap.get("startTime");
            endTime = timeRangeMap.get("endTime");
            String labelTypeId = "1631119596744265729";
            List<LabelEntity> labelEntities = labelEntityService.listByType(labelTypeId);
            AtomicLong total = new AtomicLong();
            String finalStartTime = startTime;
            String finalEndTime = endTime;
            labelEntities.forEach(e -> {
                CompletableFuture<CountVO> async = CompletableFuture.supplyAsync(() -> {
                    CountVO countVO = esStatisticsService.orientation(eventId, e.getId(), finalStartTime, finalEndTime, type);
                    total.addAndGet(countVO.getValue());
                    supply(countVO, finalStartTime, finalEndTime, type);
                    return countVO;
                });
                try {
                    CountVO countVO = async.get();
                    list.add(countVO);
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            });
            for (CountVO countVO : list) {
                long value = countVO.getValue();
                long totalNum = total.get();
                String divide = CalculateUtil.divide(String.valueOf(value), String.valueOf(totalNum));
                String percentage = "0%";
                if (StringUtils.isNotEmpty(divide)) {
                    percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
                }
                countVO.setPercentage(percentage);
            }
        }
        return Result.OK(list);
    }

    /**
     * 3.6 信息传播走势
     *
     * @param eventId   专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/flowData")
    public Result<?> flowData(@RequestParam String eventId,
                              @RequestParam(required = false) String startTime,
                              @RequestParam(required = false) String endTime,
                              @RequestParam Integer type) {
        Map<String, String> map = esStatisticsService.totalAndMax(eventId, null, null, type);
        String totalCount = map.get("totalCount");
        List<CountVO> list = new ArrayList<>();
        if (!totalCount.equals("0")) {
            String maxTime = map.get("maxTime");
            Event event = eventService.getById(eventId);
            if (StringUtils.isEmpty(startTime)) {
                startTime = DateUtil.formatDateTime(event.getStartTime());
            }
            Map<String, String> timeRangeMap = getTimeRange(startTime, endTime, maxTime, type);
            startTime = timeRangeMap.get("startTime");
            endTime = timeRangeMap.get("endTime");
            List<CountVO> dataList = esStatisticsService.flowData(eventId, startTime, endTime, type);
            list = supplyChildren(dataList, startTime, endTime, type);
        }
        return Result.OK(list);
    }


    /**
     * 传播路径
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/7/14
     */
    @GetMapping("/propagationPath")
    public Result<?> propagationPath(@RequestParam String eventId) {
        String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.PROPAGATION_KEY + eventId;
        Object cacheObject = redisUtil.get(key);
        if (ObjectUtils.isEmpty(cacheObject)) {
            cacheObject = analysisService.propagationPath(eventId);
        }
        return Result.OK(cacheObject);
    }

    /**
     * 按来源分析
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/origin")
    public Result<?> origin(@RequestParam String eventId,
                            @RequestParam(required = false) String startTime,
                            @RequestParam(required = false) String endTime) {
        List<CountVO> list = esStatisticsService.origin(eventId, startTime, endTime);
        return Result.OK(list);
    }


    /**
     * 影响评估
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2025/7/14
     */
    @GetMapping("/impactAssessment")
    public Result<?> impactAssessment(@RequestParam String eventId,
                                      @RequestParam(required = false) String startTime,
                                      @RequestParam(required = false) String endTime,
                                      @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.IMPACT_ASSESSMENT.getCode());
        //String result = analysisService.llmResult(eventId, startTime, endTime, AnalysisColumnEnum.IMPACT_ASSESSMENT.getCode());
        return Result.OK(JSON.parseArray(versionData,JSONObject.class));
    }

    /**
     * 驱动因素
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/drivingFactors")
    public Result<?> drivingFactors(@RequestParam String eventId,
                                    @RequestParam(required = false) String startTime,
                                    @RequestParam(required = false) String endTime,
                                    @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.DRIVING_FACTORS.getCode());
        //String result = analysisService.llmResult(eventId, startTime, endTime, AnalysisColumnEnum.DRIVING_FACTORS.getCode());
        return Result.OK(versionData);
    }

    /**
     * 相关指标
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/relatedIndicator")
    public Result<?> relatedIndicator(@RequestParam String eventId,
                                      @RequestParam(required = false) String startTime,
                                      @RequestParam(required = false) String endTime,
                                      @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.RELATED_INDICATOR.getCode());
        //String result = analysisService.llmResult(eventId, startTime, endTime, AnalysisColumnEnum.RELATED_INDICATOR.getCode());
        return Result.OK(versionData);
    }

    /**
     * 举措建议
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/measureSuggest")
    public Result<?> measureSuggest(@RequestParam String eventId,
                                    @RequestParam(required = false) String startTime,
                                    @RequestParam(required = false) String endTime,
                                    @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.MEASURE_SUGGEST.getCode());
        //String result = analysisService.llmResult(eventId, startTime, endTime, AnalysisColumnEnum.MEASURE_SUGGEST.getCode());
        return Result.OK(JSON.parseArray(versionData,JSONObject.class));
    }

    /**
     * 知识词条
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/knowledgeEntry")
    public Result<?> knowledgeEntry(@RequestParam String eventId,
                                    @RequestParam(required = false) String startTime,
                                    @RequestParam(required = false) String endTime,
                                    @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.KNOWLEDGE_ENTRY.getCode());
        //String result = analysisService.llmResult(eventId, startTime, endTime, AnalysisColumnEnum.KNOWLEDGE_ENTRY.getCode());
        return Result.OK(JSON.parseArray(versionData,JSONObject.class));
    }

    /**
     * 企业案例
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/enterpriseCase")
    public Result<?> enterpriseCase(@RequestParam String eventId,
                                    @RequestParam(required = false) String startTime,
                                    @RequestParam(required = false) String endTime,
                                    @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.ENTERPRISE_CASE.getCode());
        //String result = analysisService.llmResult(eventId, startTime, endTime, AnalysisColumnEnum.ENTERPRISE_CASE.getCode());
        return Result.OK(JSON.parseArray(versionData,JSONObject.class));
    }

    /**
     * 各国政策
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/4/12
     */
    /*@GetMapping("/countryPolicy")
    public Result<?> countryPolicy(@RequestParam String eventId,
                                   @RequestParam(required = false) String startTime,
                                   @RequestParam(required = false) String endTime,
                                   @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.COUNTRY_POLICY.getCode());
        //String result = analysisService.llmResult(eventId, startTime, endTime, AnalysisColumnEnum.COUNTRY_POLICY.getCode());
        return Result.OK(JSON.parseArray(versionData,JSONObject.class));
    }*/

    /**
     * 类似事件
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/similarEvent")
    public Result<?> similarEvent(@RequestParam String eventId,
                                          @RequestParam(required = false) String startTime,
                                          @RequestParam(required = false) String endTime,
                                          @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        List<Map<String,Object>> dataList = new ArrayList<>();
        Map<String,Object> domestic = new HashMap<>();
        String domesticSimilarEvent = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.DOMESTIC_SIMILAR_EVENT.getCode());
        domestic.put("type",AnalysisColumnEnum.DOMESTIC_SIMILAR_EVENT.getName());
        domestic.put("data",JSON.parseArray(domesticSimilarEvent,JSONObject.class));
        dataList.add(domestic);
        Map<String,Object> foreign = new HashMap<>();
        String foreignSimilarEvent = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.FOREIGN_SIMILAR_EVENT.getCode());
        foreign.put("type",AnalysisColumnEnum.FOREIGN_SIMILAR_EVENT.getName());
        foreign.put("data",JSON.parseArray(foreignSimilarEvent,JSONObject.class));
        dataList.add(foreign);
        return Result.OK(dataList);
    }

    /**
     * 中国类似事件
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/domesticSimilarEvent")
    public Result<?> domesticSimilarEvent(@RequestParam String eventId,
                                          @RequestParam(required = false) String startTime,
                                          @RequestParam(required = false) String endTime,
                                          @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.DOMESTIC_SIMILAR_EVENT.getCode());
        //String result = analysisService.llmResult(eventId, startTime, endTime, AnalysisColumnEnum.DOMESTIC_SIMILAR_EVENT.getCode());
        return Result.OK(JSON.parseArray(versionData,JSONObject.class));
    }

    /**
     * 各国类似事件
     *
     * @param eventId   事件id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/foreignSimilarEvent")
    public Result<?> foreignSimilarEvent(@RequestParam String eventId,
                                         @RequestParam(required = false) String startTime,
                                         @RequestParam(required = false) String endTime,
                                         @RequestParam(required = false) String versionId) {
        if (StringUtils.isEmpty(versionId)) {
            EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
            if (eventAnalysisVersion != null) {
                versionId = eventAnalysisVersion.getId();
            }
        }
        String versionData = eventAnalysisVersionRecordService.getVersionData(versionId, AnalysisColumnEnum.FOREIGN_SIMILAR_EVENT.getCode());
        //String result = analysisService.llmResult(eventId, startTime, endTime, AnalysisColumnEnum.FOREIGN_SIMILAR_EVENT.getCode());
        return Result.OK(JSON.parseArray(versionData,JSONObject.class));
    }


    /**
     * 关注
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/7/15
     */
    @GetMapping("/collect")
    public Result<?> collect(String eventId) {
        UserVo currentUser = UserUtil.getLoginUser();
        eventCollectMapService.save(new EventCollectMap().setEventId(eventId).setUserId(currentUser.getId()));
        return Result.OK();
    }

    /**
     * 取消关注
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/7/15
     */
    @GetMapping("/cancelCollect")
    public Result<?> cancelCollect(String eventId) {
        UserVo currentUser = UserUtil.getLoginUser();
        LambdaQueryWrapper<EventCollectMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(EventCollectMap::getEventId, eventId).eq(EventCollectMap::getUserId, currentUser.getId());
        eventCollectMapService.remove(queryWrapper);
        return Result.OK();
    }


    @PostMapping("/exportPPT")
    public void exportPPT(@RequestBody JSONObject params, HttpServletResponse response) throws IOException {
        String eventId = params.getString("eventId");
        JSONArray content = params.getJSONArray("content");
        String ppt = analysisService.exportPPT(eventId, content);
        if (ppt == null) {
            response.sendError(HttpServletResponse.SC_NOT_FOUND, "PPT文件生成失败");
            return;
        }
        commonService.downloadTemplate(response, ppt);
    }

    /**
     * 获取事件分析报告详细信息
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/reportInfo")
    public Result<?> reportInfo(@RequestParam String eventId) {
        LambdaQueryWrapper<EventAnalysisReport> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(EventAnalysisReport::getEventId, eventId);
        EventAnalysisReport one = eventAnalysisReportService.getOne(queryWrapper);
        return Result.OK(one);
    }

    /**
     * 事件分析报告编辑
     *
     * @param report 分析报告信息
     * @author lkg
     * @date 2024/4/12
     */
    @PostMapping("/reportEdit")
    public Result<?> edit(@RequestBody EventAnalysisReport report) {
        eventAnalysisReportService.edit(report);
        return Result.OK();
    }


    /**
     * 获取趋势图数据的实际时间范围
     *
     * @param startTime 专题开始时间
     * @param endTime   专题结束时间
     * @param maxTime   峰值所在时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/4/11
     */
    private Map<String, String> getTimeRange(String startTime, String endTime, String maxTime, Integer type) {
        Map<String, String> map = new HashMap<>();
        if (StringUtils.isEmpty(endTime)) {
            endTime = DateUtil.now();
        }
        if (type == 1) {
            DateTime beginOfDay = DateUtil.beginOfDay(DateUtil.parse(maxTime, "yyyy-MM-dd HH"));
            startTime = DateUtil.formatDateTime(beginOfDay);
            DateTime endOfDay = DateUtil.endOfDay(DateUtil.parse(maxTime, "yyyy-MM-dd HH"));
            endTime = DateUtil.formatDateTime(endOfDay);
        } else if (type == 2) {
            long between = DateUtil.betweenDay(DateUtil.parseDateTime(startTime), DateUtil.parseDateTime(endTime), true);
            if (between > 14) {
                DateTime startDate = DateUtil.offsetDay(DateUtil.parseDate(maxTime), -7);
                if (startDate.compareTo(DateUtil.parseDateTime(startTime)) > 0) {
                    startTime = DateUtil.formatDateTime(startDate);
                }
                DateTime endDate = DateUtil.offsetDay(DateUtil.parseDate(maxTime), 7);
                if (endDate.compareTo(DateUtil.parseDateTime(endTime)) < 0) {
                    endTime = DateUtil.formatDateTime(endDate);
                }
            }
        }
        map.put("startTime", startTime);
        map.put("endTime", endTime);
        return map;
    }


    //补充缺失的时间
    private void supply(CountVO countVO, String startTime, String endTime, Integer type) {
        List<CountVO> children = countVO.getChildren();
        List<CountVO> list = supplyChildren(children, startTime, endTime, type);
        countVO.setChildren(list);
    }

    //补充缺失的时间
    private List<CountVO> supplyChildren(List<CountVO> children, String startTime, String endTime, Integer type) {
        List<CountVO> list = new ArrayList<>();
        Map<String, CountVO> map = children.stream().collect(Collectors.toMap(CountVO::getName, item -> item, (k1, k2) -> k2));
        DateTime startDate = DateUtil.parseDateTime(startTime);
        DateTime endDate = DateUtil.parseDateTime(endTime);
        List<DateTime> rangeToList = new ArrayList<>();
        String format = null;
        if (type == 1) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.HOUR_OF_DAY);
            format = "yyyy-MM-dd HH";
        } else if (type == 2) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.DAY_OF_YEAR);
            format = "yyyy-MM-dd";
        }
        for (DateTime dateTime : rangeToList) {
            String date = DateUtil.format(dateTime, format);
            if (map.containsKey(date)) {
                list.add(map.get(date));
            } else {
                CountVO vo = new CountVO();
                vo.setName(date);
                vo.setValue(0L);
                list.add(vo);
            }
        }
        return list;
    }
}
