package com.zzsn.event.util;


import com.obs.services.ObsClient;
import com.obs.services.model.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * Description: obs桶文件操作
 * Author: EDY
 * Date: 2023/10/9
 */
@Component
public class ObsUtil {

    @Autowired
    ObsClient obsClient;
    /**
     * 桶名称
     */
    private final String bucketName = "zzsn";

    /**
     * 判断桶是否存在
     */
    public Boolean existsBucket(String bucket) {
        return obsClient.headBucket(bucket);
    }

    public Boolean existsBucket() {
        return obsClient.headBucket(bucketName);
    }

    /**
     * 创建文件夹本质上来说是创建了一个大小为0且对象名以“/”结尾的对象。
     * 多级文件夹创建最后一级即可，比如src1/src2/src3/，创建src1/src2/src3/即可，无需创建src1/、src1/src2/。
     * keySuffixWithSlash为文件夹名称，以 / 结尾
     */
    public boolean mkdir(String keySuffixWithSlash) {
        PutObjectResult putObjectResult = obsClient.putObject(bucketName, keySuffixWithSlash, new ByteArrayInputStream(new byte[0]));
        return putObjectResult.getStatusCode() == 200;
    }

    /**
     * 查询桶内文件夹下所有文件
     * folderPrefix 为文件夹名称，以 / 结尾
     */
    public List<ObsObject> getPathFileList(String folderPrefix) {
        ListObjectsRequest request = new ListObjectsRequest(bucketName);
        request.setPrefix(folderPrefix);
        ObjectListing result = obsClient.listObjects(request);
        return new ArrayList<>(result.getObjects());
    }


    /**
     * 删除桶内文件
     * objectKey为文件路径，起始为桶内某文件夹，或者直接为桶内文件
     */
    public boolean delFile(String objectKey) {
        DeleteObjectResult deleteObjectResult = obsClient.deleteObject(bucketName, objectKey);
        return deleteObjectResult.getStatusCode() == 200;
    }

    /**
     * 文件上传
     * objectKey为文件路径
     */
    public PutObjectResult uploadFile(String objectKey, byte[] bytes) {
        return obsClient.putObject(bucketName, objectKey, new ByteArrayInputStream(bytes));
    }

    /**
     * 文件上传
     * objectKey为文件路径
     */
    public PutObjectResult uploadFile(String objectKey, InputStream inputStream) {
        return obsClient.putObject(bucketName, objectKey, inputStream);
    }

    /**
     * 获取文件流
     */
    public InputStream getObjectStream(String objectKey) {
        ObsObject obsObject = obsClient.getObject(bucketName, objectKey);
        return obsObject.getObjectContent();
    }

    /**
     * 获取文件流
     */
    public byte[] getObjectByte(String objectKey) {
        ObsObject obsObject = obsClient.getObject(bucketName, objectKey);
        // 获取文件的输入流
        InputStream objectContent = obsObject.getObjectContent();

        // 将输入流转换为byte[]
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        byte[] buffer = new byte[4096];
        int bytesRead;
        while (true) {
            try {
                if ((bytesRead = objectContent.read(buffer)) == -1) {
                    break;
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            byteArrayOutputStream.write(buffer, 0, bytesRead);
        }
        return byteArrayOutputStream.toByteArray();
    }
}
