package com.zzsn.event.task;

import cn.hutool.core.date.DateUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.entity.EventNetwork;
import com.zzsn.event.llm.LlmService;
import com.zzsn.event.service.EventNetworkService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * 网络事件处理任务
 *
 * @author lkg
 * @date 2025/7/3
 */
@Slf4j
@Component
public class NetWorkEventTask {

    @Autowired
    private EventNetworkService eventNetworkService;
    @Autowired
    private LlmService llmService;

    @Value("${scheduling.yjzxEnable:false}")
    Boolean yjzxEnable;
    @Value("${python.hot-crawler:}")
    private String hotCrawlerPath;

    //final static String PROMPT = "根据提供的网络事件对象集合json字符串中事件标题-【title】字段，判断是否为政治经济领域相关的事件，并按以下格式返回符合条件的网络事件集合：[{\"title\": \"政治\"}]";
    final static String PROMPT = "你是一个网络舆情事件筛选专员。擅长从网络事件对象集合中精准筛选出经济领域相关事件以及会影响经济发展的政治事件。\n" +
            "1. 依据提供的网络事件对象集合json字符串中的事件标题-【title】字段，精准判断事件是否属于经济领域相关事件或会影响经济发展的政治事件。\n" +
            "2. 按照指定格式返回符合经济领域条件的网络事件集合，且无需额外解释。\n" +
            "3. 必须严格依据事件标题-【title】字段进行判断。\n" +
            "4. 禁止返回不符合经济领域条件的事件。\n" +
            "5. 严格按照以下json格式返回数据:\n" +
            "[\n" +
            "  {\n" +
            "    \"title\": \"经济\"\n" +
            "  }\n" +
            "]";

    /**
     * 定时拉取热榜数据 (1小时一次)
     *
     * @author lkg
     * @date 2025/7/4
     */
    @Scheduled(cron = "0 0 0/1 * * ?")
    public void execute() {
        for (int i = 1; i < 3; i++) {
            int finalI = i;
            CompletableFuture.runAsync(() -> handler(finalI));
        }
    }

    public void test(Integer type) {
        List<EventNetwork> networkList = getNetWordEventList(type);
        String response = llmService.model(null,null, PROMPT, JSONObject.toJSONString(networkList));
        if (response.contains("```json")) {
            response = response.substring(response.indexOf("```json") + 7, response.lastIndexOf("```"));
        }
        List<EventNetwork> hitList = JSON.parseArray(response, EventNetwork.class);
        System.out.println(JSON.toJSONString(hitList));
    }


    private void handler(Integer type) {
        List<EventNetwork> networkList = getNetWordEventList(type);
        if (CollectionUtils.isEmpty(networkList)) {
            return;
        }
        String today = DateUtil.today();
        LambdaQueryWrapper<EventNetwork> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(EventNetwork::getPublishDate, today).eq(EventNetwork::getType, type).eq(EventNetwork::getLatest, 1);
        List<EventNetwork> oldList = eventNetworkService.list(queryWrapper);
        //最终要处理的数据 过模型判断是否 是所需要的事件[政治经济领域相关]
        List<EventNetwork> finalList;
        if (CollectionUtils.isEmpty(oldList)) {
            finalList = networkList;
        } else {
            finalList = networkList.stream().filter(network -> oldList.stream()
                            .noneMatch(old -> old.getTitle().equals(network.getTitle())))
                    .collect(Collectors.toList());
            //变更状态 latest = 0，不是最新
            List<EventNetwork> removedList = oldList.stream().filter(network -> networkList.stream()
                            .noneMatch(fresh -> fresh.getTitle().equals(network.getTitle())))
                    .collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(removedList)) {
                List<EventNetwork> collect = removedList.stream().filter(network -> network.getYnNeed() == 0).collect(Collectors.toList());
                collect.forEach(network -> network.setLatest(0));
                eventNetworkService.updateBatchById(collect);
            }
        }
        if (CollectionUtils.isNotEmpty(finalList)) {
            finalList.forEach(network -> network.setLatest(1));
            String response = llmService.model(null,null, PROMPT, JSONObject.toJSONString(finalList));
            if (StringUtils.isNotEmpty(response)) {
                try {
                    if (response.contains("```json")) {
                        response = response.substring(response.indexOf("```json") + 7, response.lastIndexOf("```"));
                    }
                    List<EventNetwork> hitList = JSON.parseArray(response, EventNetwork.class);
                    finalList.forEach(network -> {
                        List<String> titleLis = hitList.stream().map(EventNetwork::getTitle).collect(Collectors.toList());
                        if (titleLis.contains(network.getTitle())) {
                            network.setYnNeed(1);
                        }
                    });
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            eventNetworkService.saveOrUpdateBatch(finalList);
        }
        log.info("{}-网络事件数据采集更新完成", type == 1 ? "百度热榜" : "新浪热榜");
    }


    /**
     * 调用采集接口，获取网络事件
     *
     * @param type 来源类型(1-百度;2-新浪新闻)
     * @author lkg
     * @date 2025/7/4
     */
    private List<EventNetwork> getNetWordEventList(Integer type) {
        try {
            String option = HttpUtil.get(hotCrawlerPath.replace("OPTION", type.toString()));
            if (StringUtils.isNotEmpty(option)) {
                List<EventNetwork> eventNetworks = JSON.parseArray(option, EventNetwork.class);
                eventNetworks.forEach(network -> network.setType(type));
                return eventNetworks;
            }
        } catch (Exception e) {
            log.info("{}-数据采集异常", type == 1 ? "百度热榜" : "新浪热榜");
            e.printStackTrace();
        }
        return new ArrayList<>();
    }
}
