package com.zzsn.event.controller;

import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.clb.common.model.task.dto.titr.KeyWordsDTO;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.KeyWords;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.IKeyWordsService;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.entity.XxlJobInfo;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * 平台事件管理页
 *
 * @author lkg
 * @date 2024/5/7
 */
@RestController
@RequestMapping("/manage")
public class PlatEventManageController {


    @Autowired
    private IEventService eventService;
    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;
    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private RedisUtil redisUtil;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;


    /**
     * 分页列表-新平台管理
     *
     * @param subjectCondition 筛选条件
     * @param pageNo           当前页
     * @param pageSize         返回条数
     * @author lkg
     * @date 2024/4/28
     */
    @GetMapping("/pageList")
    public Result<?> newPlatPageList(SubjectCondition subjectCondition,
                                     @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                     @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        Page<EventNewPlatVO> pageList = eventService.newPlatPageList(subjectCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 1.2 添加
     *
     * @param eventParam
     * @return
     */
    @PostMapping(value = "/add")
    public Result<?> add(@RequestBody AddEventParam eventParam) {
        KeywordsVO keywordsVO = eventParam.getKeywordsVO();
        if (keywordsVO != null) {
            UserVo currentUser = UserUtil.getLoginUser();
            eventParam.setCreateBy(currentUser.getUsername());
            Event event = eventService.saveMain(eventParam);
            //新增关键词
            KeyWords keyWords = keyWordsService.saveKeyword(event, keywordsVO.getKeyword(), keywordsVO.getExclusionWord());
            CompletableFuture.runAsync(() -> {
                //插入xxlJob
                iXxlJobInfoService.eventInsert(event);
                //关键词入缓存
                KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                BeanUtils.copyProperties(keyWords, redisKeywordDTO);
                redisKeywordDTO.setStartTime(event.getStartTime());
                redisKeywordDTO.setEndTime(event.getEndTime());
                redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWords.getWordsCode(), redisKeywordDTO);
                //插入xxljob
                iXxlJobInfoService.keyWordsInsert(redisKeywordDTO);
                //为了立即响应，关键词新增时放入消息队列
                kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
                kafkaTemplate.send(Constants.EVENT_SUBJECT_MODEL, event.getEventCode());

            });
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 1.3 编辑
     *
     * @param addEventParam
     * @return
     */
    @PostMapping(value = "/edit")
    public Result<?> edit(@RequestBody AddEventParam addEventParam) {
        //关键词
        KeywordsVO keywordsVO = addEventParam.getKeywordsVO();
        if (keywordsVO != null) {
            UserVo currentUser = UserUtil.getLoginUser();
            addEventParam.setUpdateTime(new Date());
            addEventParam.setUpdateBy(currentUser.getUsername());
            eventService.updateMain(addEventParam);
            keyWordsService.update(Wrappers.<KeyWords>lambdaUpdate().eq(KeyWords::getId, keywordsVO.getId())
                    .set(KeyWords::getKeyWord, keywordsVO.getKeyword())
                    .set(KeyWords::getExclusionWord, keywordsVO.getExclusionWord()));
            CompletableFuture.runAsync(() -> {
                KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                BeanUtils.copyProperties(keywordsVO, redisKeywordDTO);
                redisKeywordDTO.setKeyWord(keywordsVO.getKeyword());
                redisKeywordDTO.setStartTime(addEventParam.getStartTime());
                redisKeywordDTO.setEndTime(addEventParam.getEndTime());
                redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keywordsVO.getWordsCode(), redisKeywordDTO);
                //为了立即响应，关键词编辑时放入消息队列
                kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
            });
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 启用
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/9/12
     */
    @GetMapping("/enable")
    public Result<?> enable(@RequestParam String eventId) {
        updateStatus(eventId, 1);
        return Result.OK();
    }

    /**
     * 启用
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/9/12
     */
    @GetMapping("/disable")
    public Result<?> disable(@RequestParam String eventId) {
        updateStatus(eventId, 0);
        return Result.OK();
    }


    /**
     * 通过id删除
     *
     * @param id 事件id
     * @return
     */
    @GetMapping(value = "/delete")
    public Result<?> delete(@RequestParam(name = "id") String id) {
        eventService.deleteMain(id);
        CompletableFuture.runAsync(() -> {
            EventVO eventVO = eventService.queryInfo(id);
            iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
            //删除关键词
            KeywordsVO keywordsVO = eventVO.getKeywordsVO();
            if (keywordsVO != null) {
                String wordsCode = keywordsVO.getWordsCode();
                keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + wordsCode);
                iXxlJobInfoService.deleteByInfosourceCode(wordsCode);
                //todo 是否删除对应的资讯数据
            }
        });
        return Result.OK();
    }

    /**
     * 批量删除
     *
     * @param ids 事件id，多个用逗号隔开
     * @author lkg
     * @date 2024/12/17
     */
    @GetMapping(value = "/deleteBatch")
    public Result<?> deleteBatch(@RequestParam(name = "ids") String ids) {
        String[] idArr = ids.split(",");
        //删除
        for (String id : idArr) {
            eventService.deleteMain(id);
        }
        //异步删除关联关系
        CompletableFuture.runAsync(() -> {
            for (String id : idArr) {
                EventVO eventVO = eventService.queryInfo(id);
                iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
                //删除关键词
                KeywordsVO keywordsVO = eventVO.getKeywordsVO();
                if (keywordsVO != null) {
                    String wordsCode = keywordsVO.getWordsCode();
                    keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                    redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + wordsCode);
                    iXxlJobInfoService.deleteByInfosourceCode(wordsCode);
                }
            }
        });
        //todo 是否删除对应的资讯数据
        return Result.OK();
    }


    private void updateStatus(String eventId, Integer status) {
        eventService.update(Wrappers.<Event>lambdaUpdate().eq(Event::getId, eventId).set(Event::getStatus, status));
        CompletableFuture.runAsync(() -> {
            Event event = eventService.getById(eventId);
            iXxlJobInfoService.update(Wrappers.<XxlJobInfo>lambdaUpdate().eq(XxlJobInfo::getInfoSourceCode, event.getEventCode())
                    .set(XxlJobInfo::getTriggerStatus, status));
            //关键词
            KeywordsVO keywordsVO = keyWordsService.keywordInfoByEventId(eventId);
            iXxlJobInfoService.update(Wrappers.<XxlJobInfo>lambdaUpdate().eq(XxlJobInfo::getInfoSourceCode, keywordsVO.getWordsCode())
                    .set(XxlJobInfo::getTriggerStatus, status));
            if (1 == status) {
                kafkaTemplate.send(Constants.EVENT_SUBJECT_MODEL, event.getEventCode());
            }
        });
    }


    /**
     * @param event 条件封装
     * @描述 查询事件列表
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/listAll")
    public Result<?> listAll(@RequestBody Event event) {
        LambdaQueryWrapper<Event> queryWrapper = Wrappers.lambdaQuery();
        setListAllQueryWrapper(event, queryWrapper);
        List<Event> list = eventService.list(queryWrapper);
        return Result.OK(list);
    }

    /**
     * @param event    条件封装
     * @param pageNo
     * @param pageSize
     * @描述 查询事件列表（分页，产业链关联事件时查询用）
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/pageListAll")
    public Result<?> pageListAll(@RequestBody Event event, @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        LambdaQueryWrapper<Event> queryWrapper = Wrappers.lambdaQuery();
        setListAllQueryWrapper(event, queryWrapper);
        Page<Event> page = new Page<>(pageNo, pageSize);
        IPage<Event> pageList = eventService.page(page, queryWrapper);
        return Result.OK(pageList);
    }
    private void setListAllQueryWrapper(@RequestBody Event event, LambdaQueryWrapper<Event> queryWrapper) {
        String eventType = event.getEventType();
        String startDate = event.getStartDate();
        String endDate = event.getEndDate();
        String keyword = event.getKeyword();
        queryWrapper.eq(StringUtils.isNotEmpty(eventType), Event::getEventType, eventType);
        if (StringUtils.isNotEmpty(startDate)) {
            queryWrapper.ge(Event::getCreateTime, startDate);
        }
        if (StringUtils.isNotEmpty(endDate)) {
            queryWrapper.le(Event::getCreateTime, endDate);
        }
        if (StringUtils.isNotEmpty(keyword)) {
            queryWrapper.and(q -> {
                q.eq(Event::getEventType, keyword).or().like(Event::getEventName, keyword).or().eq(Event::getEventDescribe, keyword);
            });
        }
    }
}
