package com.zzsn.event.controller.reportData;

import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.service.CommonService;
import com.zzsn.event.service.SubjectService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.util.EsIndexUtil;
import com.zzsn.event.util.FieldUtil;
import com.zzsn.event.vo.es.Label;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.index.query.TermsQueryBuilder;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 专题报告数据接口
 *
 * @author lkg
 * @date 2025/4/9
 */
@RestController
@RequestMapping("/report/data/api")
public class ReportDataController {

    @Resource
    private RestHighLevelClient client;
    @Autowired
    private SubjectService subjectService;
    @Autowired
    private CommonService commonService;

    /**
     * 检索字段集合
     *
     * @author lkg
     * @date 2025/4/9
     */
    @GetMapping("/searchFieldList")
    public Result<?> fieldList() {
        List<Map<String, Object>> fieldDetails = FieldUtil.requestFieldDetails(SearchCondition.class);
        return Result.OK(fieldDetails);
    }

    /**
     * 响应字段集合
     *
     * @author lkg
     * @date 2025/4/9
     */
    @GetMapping("/responseFieldList")
    public Result<?> responseFieldList() {
        List<Map<String, Object>> fieldDetails = FieldUtil.responseFieldDetails(ReportDataVO.class);
        return Result.OK(fieldDetails);
    }

    /**
     * 专题资讯列表
     *
     * @author lkg
     * @date 2025/4/9
     */
    @PostMapping("/dataList")
    public Result<?> dataList(@RequestBody SearchCondition searchCondition) {
        try {
            List<ReportDataVO> pageList = pageList(searchCondition);
            return Result.OK(pageList);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return Result.OK();
    }


    /**
     * 事件对应专题库的资讯分页列表
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2024/5/6
     */
    public List<ReportDataVO> pageList(SearchCondition searchCondition) throws IOException {
        List<String> subjectIdList = new ArrayList<>();
        subjectIdList.add(searchCondition.getSubjectId());
        String minDate = subjectService.getMinCreateTime(subjectIdList);
        String[] indexArr = EsIndexUtil.getIndexIntervalYearStr(Constants.SUBJECT_INDEX, minDate);
        SearchRequest searchRequest = new SearchRequest(indexArr);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        String[] fetchFields = searchCondition.getFetchFields();
        String[] excludeFields = searchCondition.getExcludeFields();
        if (fetchFields != null && fetchFields.length > 0) {
            searchSourceBuilder.fetchSource(fetchFields, null);
        } else if (excludeFields != null && excludeFields.length > 0) {
            searchSourceBuilder.fetchSource(null, excludeFields);
        }
        //设置分页参数
        Integer pageNo = searchCondition.getPageNo();
        Integer pageSize = searchCondition.getPageSize();
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        searchSourceBuilder.size(pageSize);
        //排序规则
        String column = searchCondition.getColumn();
        String order = searchCondition.getOrder();
        if (searchCondition.getTopSortValid() == 1) {
            searchSourceBuilder.sort("topNum", SortOrder.DESC);
        }
        switch (column) {
            case "score":
                if (order.equals("asc")) {
                    searchSourceBuilder.sort("score", SortOrder.ASC);
                    searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                } else if (order.equals("desc")) {
                    searchSourceBuilder.sort("score", SortOrder.DESC);
                    searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                }
                break;
            case "publishDate":
                if (order.equals("desc")) {
                    searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                    searchSourceBuilder.sort("score", SortOrder.DESC);
                } else if (order.equals("asc")) {
                    searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                    searchSourceBuilder.sort("score", SortOrder.ASC);
                }
                break;
            default:
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                break;
        }
        //审核状态
        Integer status = searchCondition.getStatus();
        if (status != null) {
            if (status == 1) {
                searchCondition.setCheckStatus(1);
            } else if (status == 2) {
                searchCondition.setCheckStatus(0);
                searchCondition.setDeleteFlag(0);
            } else if (status == 3) {
                searchCondition.setCheckStatus(3);
            } else if (status == 4) {
                searchCondition.setDeleteFlag(1);
            } else if (status == 5) {
                searchCondition.setIsFreeCheck(1);
            }
        }
        //构建es查询条件
        BoolQueryBuilder boolQuery = buildQuery(searchCondition);
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHit[] searchHits = searchResponse.getHits().getHits();
        List<ReportDataVO> list = new ArrayList<>();
        for (SearchHit hit : searchHits) {
            String queryInfo = hit.getSourceAsString();
            ReportDataVO info = JSONUtil.toBean(queryInfo, ReportDataVO.class);
            info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
            info.setCreateDate(EsDateUtil.esFieldDateMapping(info.getCreateDate()));
            list.add(info);
        }
        return list;
    }

    /**
     * 构建通用的es查询语句
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2024/12/25
     */
    private BoolQueryBuilder buildQuery(SearchCondition searchCondition) {
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        String subjectId = searchCondition.getSubjectId();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectId));
        String searchScope = searchCondition.getSearchScope();
        String searchWord = searchCondition.getSearchWord();
        if (StringUtils.isNotEmpty(searchWord)) {
            if ("all".equalsIgnoreCase(searchScope)) {
                boolQuery.must(QueryBuilders.multiMatchQuery(searchWord, "title", "content", "origin"));
            } else {
                boolQuery.must(QueryBuilders.matchPhraseQuery(searchScope, searchWord));
            }
        }
        Integer checkStatus = searchCondition.getCheckStatus();
        Integer deleteFlag = searchCondition.getDeleteFlag();
        if (checkStatus != null) {
            if (checkStatus == 0) {
                BoolQueryBuilder checkStatusBoolQuery = QueryBuilders.boolQuery();
                checkStatusBoolQuery.should(QueryBuilders.termQuery("checkStatus", checkStatus));
                checkStatusBoolQuery.should(QueryBuilders.boolQuery().mustNot(QueryBuilders.existsQuery("checkStatus")));
                boolQuery.must(checkStatusBoolQuery);
            } else {
                boolQuery.must(QueryBuilders.termQuery("checkStatus", checkStatus));
            }
        } else {
            if (deleteFlag == null) {
                deleteFlag = 0;
            }
        }
        //删除状态查询
        if (deleteFlag != null) {
            if (deleteFlag != 0) {
                boolQuery.must(QueryBuilders.termQuery("deleteFlag", "1"));
            } else {
                boolQuery.mustNot(QueryBuilders.termQuery("deleteFlag", "1"));
            }
        }
        //免审核状态
        if (searchCondition.getIsFreeCheck() != null) {
            boolQuery.must(QueryBuilders.termQuery("isFreeCheck", searchCondition.getClassificationType()));
        }
        boolQuery.mustNot(QueryBuilders.matchQuery("type", "video"));
        //专题库类别筛选
        if (CollectionUtils.isNotEmpty(searchCondition.getClassificationType())) {
            boolQuery.must(QueryBuilders.termsQuery("classificationType", searchCondition.getClassificationType()));
        }
        //是否原创
        if (StringUtils.isNotEmpty(searchCondition.getOriginality())) {
            boolQuery.must(QueryBuilders.matchQuery("originality", searchCondition.getOriginality()));
        }
        //得分范围筛选
        Integer minScore = searchCondition.getMinScore();
        if (minScore != null) {
            boolQuery.filter(QueryBuilders.rangeQuery("score").gte(minScore));
        }
        Integer maxScore = searchCondition.getMaxScore();
        if (maxScore != null) {
            boolQuery.filter(QueryBuilders.rangeQuery("score").lte(maxScore));
        }
        //时间过滤筛选-前端传参
        if (searchCondition.getStartTime() != null) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(searchCondition.getStartTime()))));
        }
        if (searchCondition.getEndTime() != null) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(searchCondition.getEndTime()))));
        } else {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(new Date()))));
        }
        //关联标签id集合
        List<String> labelIds = searchCondition.getLabelIds();
        if (CollectionUtils.isNotEmpty(labelIds)) {
            Set<String> relationIdSet = new HashSet<>(labelIds);
            TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", relationIdSet);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        //正面标签
        String tendencyLabelId = searchCondition.getTendencyLabelId();
        if (StringUtils.isNotBlank(tendencyLabelId)) {
            TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationId", tendencyLabelId);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        //地域标签
        List<String> areaLabelIds = searchCondition.getAreaLabelIds();
        if (CollectionUtils.isNotEmpty(areaLabelIds)) {
            Set<String> relationIdSet = new HashSet<>(areaLabelIds);
            TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", relationIdSet);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        //企业标签
        List<String> enterpriseLabelTypeIds = searchCondition.getEnterpriseLabelTypeIds();
        List<String> socialCreditCodeList = searchCondition.getSocialCreditCodeList();
        if (CollectionUtils.isNotEmpty(socialCreditCodeList) || CollectionUtils.isNotEmpty(enterpriseLabelTypeIds)) {
            if (CollectionUtils.isEmpty(socialCreditCodeList)) {
                socialCreditCodeList = commonService.codesByLabels(enterpriseLabelTypeIds);
            }
            Set<String> codeSet = new HashSet<>(socialCreditCodeList);
            TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", codeSet);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        //信息源标签
        List<Label> labelList = searchCondition.getInfoSourceLabelList();
        if (CollectionUtils.isNotEmpty(labelList)) {
            List<String> collect = labelList.stream().map(label -> label.getLabelMark() + "-" + label.getRelationId()).collect(Collectors.toList());
            TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", collect);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        return boolQuery;
    }
}
