package com.zzsn.event.controller.thirdApi;

import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.clb.common.model.task.dto.titr.KeyWordsDTO;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.*;
import com.zzsn.event.enums.CodePrefixEnum;
import com.zzsn.event.service.*;
import com.zzsn.event.util.CodeGenerateUtil;
import com.zzsn.event.util.CronUtil;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.vo.EventVO;
import com.zzsn.event.vo.KeywordsVO;
import com.zzsn.event.vo.RegionVO;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 *
 *
 * @author lkg
 * @date 2024/10/12
 */
@Slf4j
@RestController
@RequestMapping("/third/api")
public class EventApi {

    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;
    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private RedisUtil redisUtil;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private IEventService eventService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;
    @Autowired
    private IProjectSubjectMapService projectSubjectMapService;
    @Autowired
    private IEventTagService eventTagService;
    @Autowired
    private EventRegionMapService eventRegionMapService;

    /**
     * 新增事件(页面简化版)
     *
     * @param addEventVO 事件信息
     * @author lkg
     * @date 2024/7/22
     */
    @PostMapping("/saveEvent")
    public Result<?> addEvent(@RequestBody ThirdAddEventVO addEventVO) {
        KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
        if (keywordsVO != null) {
            Event event = simpleSave(addEventVO);
            KeyWords keyWords = keyWordsService.saveKeyword(event, keywordsVO.getKeyword(), keywordsVO.getExclusionWord());
            event.setKeyWords(keyWords);
            CompletableFuture.runAsync(()->{
                iXxlJobInfoService.eventInsert(event);
                //关键词
                KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                BeanUtils.copyProperties(keyWords, redisKeywordDTO);
                redisKeywordDTO.setStartTime(event.getStartTime());
                redisKeywordDTO.setEndTime(event.getEndTime());
                redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWords.getWordsCode(), redisKeywordDTO);
                //插入xxljob
                iXxlJobInfoService.keyWordsInsert(redisKeywordDTO);
                //为了立即响应，关键词新增时放入一个首次录入消息队列
                kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
                kafkaTemplate.send(Constants.EVENT_SUBJECT_MODEL, event.getEventCode());
            });
            return Result.OK(event);
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 编辑事件
     *
     * @param addEventVO 事件信息
     * @author lkg
     * @date 2024/11/11
     */
    @PostMapping("/editEvent")
    public Result<?> editEvent(@RequestBody ThirdAddEventVO addEventVO) {
        KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
        if (keywordsVO != null) {
            simpleUpdate(addEventVO);
            keyWordsService.update(Wrappers.<KeyWords>lambdaUpdate().eq(KeyWords::getId, keywordsVO.getId())
                    .set(KeyWords::getKeyWord, keywordsVO.getKeyword())
                    .set(KeyWords::getExclusionWord, keywordsVO.getExclusionWord()));
            CompletableFuture.runAsync(()->{
                //关键词
                KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                BeanUtils.copyProperties(keywordsVO, redisKeywordDTO);
                redisKeywordDTO.setKeyWord(keywordsVO.getKeyword());
                redisKeywordDTO.setStartTime(addEventVO.getStartTime());
                redisKeywordDTO.setEndTime(addEventVO.getEndTime());
                redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keywordsVO.getWordsCode(), redisKeywordDTO);
                //为了立即响应，关键词新增时放入一个首次录入消息队列
                kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
            });
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 通过id删除
     *
     * @param id 事件id
     * @return
     */
    @ApiOperation(value = "事件-通过id删除", notes = "事件-通过id删除")
    @GetMapping(value = "/deleteEvent")
    public Result<?> delete(@RequestParam(name = "id") String id) {
        EventVO eventVO = eventService.queryInfo(id);
        eventService.deleteMain(id);
        CompletableFuture.runAsync(() -> {
            iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
            KeywordsVO keywordsVO = eventVO.getKeywordsVO();
            if (keywordsVO != null) {
                String wordsCode = keywordsVO.getWordsCode();
                keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + wordsCode);
                iXxlJobInfoService.deleteByInfosourceCode(wordsCode);
            }
        });
        return Result.OK();
    }

    public void simpleUpdate(ThirdAddEventVO addEventVO){
        String eventId = addEventVO.getId();
        Event event = new Event();
        BeanUtils.copyProperties(addEventVO, event);
        event.setUpdateBy(addEventVO.getUsername());
        event.setUpdateTime(new Date());
        eventService.updateById(event);
        CompletableFuture.runAsync(()->{
            //地域关系绑定
            List<RegionVO> regionList = addEventVO.getRegionList();
            addRegionMap(eventId, regionList);
        });
    }

    @Transactional
    public Event simpleSave(ThirdAddEventVO addEventVO) {
        Event event = new Event();
        BeanUtils.copyProperties(addEventVO, event);
        event.setStatus(1);
        Date endTime = event.getEndTime();
        if (endTime == null) {
            endTime = DateUtil.addDate(new Date(), 15);
        }
        event.setEndTime(endTime);
        event.setIncreAnaRule(20);
        event.setTotalAnaRule(50);
        event.setTimeAnaRule(5);
        String subjectCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.EVENT_DEFAULT.getValue());
        event.setEventCode(subjectCode);
        //默认发布
        event.setPublishStatus(1);
        event.setPublishDate(DateUtil.dateToString(new Date()));
        //调度周期，默认2小时
        event.setUnit("2");
        event.setSpace(1);
        event.setCron(CronUtil.generateCron("2",1));
        event.setCreateBy(addEventVO.getUsername());
        eventService.save(event);
        String eventId = event.getId();
        CompletableFuture.runAsync(()->{
            saveMapMain(eventId);
            //事件标签
            eventTagService.save(EventTag.builder().eventId(eventId).build());
            //地域关系绑定
            List<RegionVO> regionList = addEventVO.getRegionList();
            addRegionMap(eventId, regionList);
            //默认绑定搜索引擎、模型
            eventService.saveDefaultMap(eventId);
        });
        return event;
    }

    private void saveMapMain(String eventId) {
        //专题和分类关系
        SubjectTypeMap subjectTypeMap = new SubjectTypeMap();
        subjectTypeMap.setSubjectId(eventId);
        //分类默认 研究中心
        subjectTypeMap.setTypeId("1802619433932296193");
        subjectTypeMapService.save(subjectTypeMap);
        //专题和项目关系
        ProjectSubjectMap projectSubjectMap = new ProjectSubjectMap();
        //项目默认 研究中心
        projectSubjectMap.setProjectId("1476527644425682945");
        projectSubjectMap.setSubjectId(eventId);
        projectSubjectMapService.save(projectSubjectMap);
    }



    private void addRegionMap(String eventId, List<RegionVO> regionList) {
        if (CollectionUtils.isNotEmpty(regionList)) {
            List<EventRegionMap> dataList = new ArrayList<>();
            for (RegionVO regionVO : regionList) {
                EventRegionMap eventRegionMap = new EventRegionMap();
                eventRegionMap.setEventId(eventId);
                eventRegionMap.setRegionId(regionVO.getId());
                eventRegionMap.setTopRegionId(regionVO.getTopId());
                eventRegionMap.setType(regionVO.getType());
                dataList.add(eventRegionMap);
            }
            eventRegionMapService.saveBatch(dataList);
        }
    }
}
