package com.zzsn.event.util.ppt;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;

import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * ppt工具类
 *
 * @author lkg
 * @date 2025/7/19
 */
@Slf4j
public class PPTUtil {

    //讯飞-智能PPT
    public static final String baseUrl = "https://zwapi.xfyun.cn/api/ppt/v2";
    //默认模板id
    public static final String templateId = "20240807EA3CB30";

    private final static OkHttpClient client = new OkHttpClient().newBuilder()
            .connectionPool(new ConnectionPool(100, 5, TimeUnit.MINUTES))
            .readTimeout(60 * 10, TimeUnit.SECONDS)
            .build();

    /**
     * 生成大纲
     *
     * @param appId     应用id
     * @param timestamp 时间戳
     * @param signature 签名
     * @param query     大模型提问
     * @author lkg
     * @date 2025/7/19
     */
    public static CreateResponse createOutline(String appId, String timestamp, String signature, String query) {
        MultipartBody.Builder builder = new MultipartBody.Builder();
        builder.setType(MultipartBody.FORM);
        builder.addFormDataPart("query", query);
        RequestBody requestBody = builder.build();
        Request request = new Request.Builder().url(baseUrl + "/createOutline")
                .addHeader("appId", appId)
                .addHeader("timestamp", timestamp)
                .addHeader("signature", signature)
                .post(requestBody)
                .build();
        try (Response response = client.newCall(request).execute()) {
            if (response.isSuccessful()) {
                String string = Objects.requireNonNull(response.body()).string();
                return JSONObject.parseObject(string, CreateResponse.class);
            } else {
                log.info("PPT大纲生成异常:{}", Objects.requireNonNull(response.body()).string());
                return null;
            }
        } catch (IOException e) {
            log.info("PPT生成进度查询异常:{}", e.getMessage());
        }
        return null;
    }

    /**
     * 根据大纲生成ppt
     *
     * @param appId     应用id
     * @param timestamp 时间戳
     * @param signature 签名
     * @param query     大模型提问
     * @param outline   ppt大纲
     * @author lkg
     * @date 2025/7/19
     */
    public static CreateResponse createByOutline(String appId, String timestamp, String signature, String query, String outline) {
        JSONObject jsonObject = new JSONObject();
        JSONObject outlineJson = JSONObject.parseObject(outline);
        jsonObject.put("outline", outlineJson);
        jsonObject.put("query", query);
        jsonObject.put("templateId", templateId);
        RequestBody requestBody = RequestBody.create(jsonObject.toString(), MediaType.get("application/json; charset=utf-8"));
        Request request = new Request.Builder()
                .url(baseUrl + "/createPptByOutline")
                .addHeader("appId", appId)
                .addHeader("timestamp", timestamp)
                .addHeader("signature", signature)
                .post(requestBody)
                .build();
        try (Response response = client.newCall(request).execute()) {
            if (response.isSuccessful()) {
                String string = Objects.requireNonNull(response.body()).string();
                return JSONObject.parseObject(string, CreateResponse.class);
            } else {
                log.info("PPT生成异常:{}", Objects.requireNonNull(response.body()).string());
                return null;
            }
        } catch (IOException e) {
            log.info("PPT生成进度查询异常:{}", e.getMessage());
        }
        return null;
    }

    /**
     * 查看ppt生成进度
     *
     * @param appId     应用id
     * @param timestamp 时间戳
     * @param signature 签名
     * @param sid       请求大纲唯一id
     * @author lkg
     * @date 2025/7/19
     */
    public static ProgressResponse checkProgress(String appId, String timestamp, String signature, String sid) {
        String url = baseUrl + "/progress?sid=" + sid;
        Request request = new Request.Builder()
                .url(url)
                .addHeader("appId", appId)
                .addHeader("timestamp", timestamp)
                .addHeader("signature", signature)
                .get()
                .build();
        try (Response response = client.newCall(request).execute()) {
            if (response.isSuccessful()) {
                String string = Objects.requireNonNull(response.body()).string();
                return JSON.parseObject(string,ProgressResponse.class);
            } else {
                log.info("PPT生成进度查询异常:{}", Objects.requireNonNull(response.body()).string());
                return null;
            }
        } catch (IOException e) {
            log.info("PPT生成进度查询异常:{}", e.getMessage());
        }
        return null;
    }
}
