package com.zzsn.event.task;

import com.zzsn.event.constant.Constants;
import com.zzsn.event.service.AnalysisService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.vo.CoOccurrenceVO;
import com.zzsn.event.vo.SubjectKafkaVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * 热词共现数据缓存
 *
 * @author lkg
 * @date 2025/7/22
 */
@Slf4j
@Component
public class CoOccurrenceTask {

    @Autowired
    private IEventService eventService;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private AnalysisService analysisService;

    /**
     * 定时缓存热词共现关系数据
     * 每天凌晨1点执行一次
     */
    @Scheduled(cron = "0 0 1 * * ?")
    public void propagationPath() {
        Date today = new Date();
        Date deadlineDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> events = eventService.eventSubjectList();
        for (SubjectKafkaVo event : events) {
            CompletableFuture.runAsync(() -> {
                String eventId = event.getId();
                String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.CO_OCCURRENCE + eventId;
                Date timeDisable = event.getTimeDisable();
                if (timeDisable != null && deadlineDate.compareTo(timeDisable) > 0) {
                    Object cacheObject = redisUtil.get(key);
                    if (cacheObject == null) {
                        List<CoOccurrenceVO> occurrenceList = analysisService.coOccurrence(eventId, null, null);
                        if (CollectionUtils.isNotEmpty(occurrenceList)) {
                            redisUtil.set(key, occurrenceList);
                            log.info("专题-{},热词共现关系数据【永久】缓存成功!", event.getSubjectName());
                        }
                    }
                } else {
                    List<CoOccurrenceVO> occurrenceList = analysisService.coOccurrence(eventId, null, null);
                    if (CollectionUtils.isNotEmpty(occurrenceList)) {
                        redisUtil.set(key, occurrenceList, 3600 * 24);
                        log.info("专题-{},热词共现关系数据缓存成功!", event.getSubjectName());
                    }
                }
            });
        }
    }
}
