package com.zzsn.event.controller.common;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.SubjectType;
import com.zzsn.event.entity.SubjectTypeMap;
import com.zzsn.event.service.ISubjectTypeMapService;
import com.zzsn.event.service.ISubjectTypeService;
import com.zzsn.event.util.tree.Node;
import com.zzsn.event.util.tree.TreeUtil;
import com.zzsn.event.vo.SubjectTreeVO;
import com.zzsn.event.vo.SubjectTypeVo;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 公共接口
 *
 * @author lkg
 * @date 2024/12/17
 */
@RestController
@RequestMapping("/type")
public class SubjectTypeController {

    @Autowired
    private ISubjectTypeService subjectTypeService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;

    /**
     * 分类列表-树型结构
     *
     * @param category 类别(1-专题;2-事件)
     * @author lkg
     * @date 2024/12/17
     */
    @GetMapping("/treeList")
    public Result<?> typeList(@RequestParam Integer category) {
        //获取当前登录用户
        List<Node> nodes = subjectTypeService.enableList(category);
        List<Node> tree = TreeUtil.tree(nodes, "0");
        return Result.OK(tree);
    }


    /**
     * 新增专题分类
     *
     * @param subjectType 专题分类
     * @author lkg
     * @date 2024/4/29
     */
    @PostMapping("/add")
    public Result<?> addSubjectType(@RequestBody SubjectType subjectType) {
        String pid = subjectType.getPid();
        if (!"0".equals(pid)) {
            int count = subjectTypeMapService.count(new LambdaQueryWrapper<SubjectTypeMap>().eq(SubjectTypeMap::getTypeId, pid));
            if (count > 0) {
                return Result.FAIL(501, "当前分类下存在专题");
            }
        }
        subjectTypeService.add(subjectType);
        return Result.OK();
    }

    /**
     * 编辑
     *
     * @param subjectType 专题分类
     * @author lkg
     * @date 2024/4/29
     */
    @PostMapping("/edit")
    public Result<?> edit(@RequestBody SubjectType subjectType){
        subjectTypeService.edit(subjectType);
        return Result.OK();
    }

    /**
     * 删除
     *
     * @param id 专题分类id
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/delete")
    public Result<?> delete(@RequestParam String id){
        int count = subjectTypeMapService.count(new LambdaQueryWrapper<SubjectTypeMap>().eq(SubjectTypeMap::getTypeId, id));
        if (count > 0) {
            return Result.FAIL(501, "当前分类下存在专题");
        }
        subjectTypeService.delete(id);
        return Result.OK();
    }

    /**
     * 专题页左侧树
     *
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/subjectAndTypeTree")
    public Result<?> subjectAndTypeTree() {
        List<SubjectTreeVO> tree = subjectTypeService.subjectAndTypeTree();
        return Result.OK(tree);
    }

    /**
     * 专题类别-通过父级id列表查询
     *
     * @return
     */
    @GetMapping(value = "/subjectAndTypeByParentId")
    public Result<?> subjectAndTypeByParentId(@RequestParam(name = "subjectType", defaultValue = "0") String subjectType,
                                    @RequestParam(name = "parentId", defaultValue = "0") String parentId) {
        List<SubjectTypeVo> allList = subjectTypeService.subjectListByType(subjectType, parentId);
        return Result.OK(allList);
    }

    /**
     * 事件页左侧树
     *
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/eventAndTypeTree")
    public Result<?> eventAndTypeTree() {
        List<SubjectTreeVO> tree = subjectTypeService.eventAndTypeTree();
        return Result.OK(tree);
    }

    /**
     * 专题类别-通过父级id列表查询
     *
     * @return
     */
    @GetMapping(value = "/eventAndTypeByParentId")
    public Result<?> eventAndTypeByParentId(@RequestParam(name = "subjectType", defaultValue = "0") String subjectType,
                                    @RequestParam(name = "parentId", defaultValue = "0") String parentId) {
        List<SubjectTypeVo> allList = subjectTypeService.eventListByType(subjectType, parentId);
        return Result.OK(allList);
    }
}
