package com.zzsn.event.controller.common;

import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Subject;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.IInfoSourceService;
import com.zzsn.event.service.SubjectService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.vo.CountVO;
import com.zzsn.event.vo.EventDataCondition;
import com.zzsn.event.vo.OverviewAnalysisVO;
import com.zzsn.event.vo.es.SpecialInformation;
import com.zzsn.event.entity.InfoSource;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 统计分析
 *
 * @author lkg
 * @date 2024/12/25
 */
@RestController
@RequestMapping("/statistical/analysis")
public class StatisticalAnalysisController {

    @Autowired
    private EsService esService;
    @Autowired
    private SubjectService subjectService;
    @Autowired
    private IInfoSourceService infoSourceService;

    /**
     * 概述分析
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/26
     */
    @GetMapping("overview")
    public Result<OverviewAnalysisVO> overviewAnalysis(@RequestParam String subjectId){
        OverviewAnalysisVO overviewAnalysisVO = new OverviewAnalysisVO();
        Subject subject = subjectService.getById(subjectId);
        overviewAnalysisVO.setSubjectName(subject.getSubjectName());
        overviewAnalysisVO.setTimeEnable(subject.getTimeEnable());
        overviewAnalysisVO.setTimeDisable(subject.getTimeDisable());
        int count = esService.count(subjectId, DateUtil.dateToString(subject.getTimeEnable()), DateUtil.dateToString(subject.getTimeDisable()));
        overviewAnalysisVO.setInfoCount(count);
        //todo  专题下绑定的信息源分类下的信息源数量
        return Result.OK(overviewAnalysisVO);
    }

    /**
     * 资讯趋势分析
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/25
     */
    @PostMapping("/dataTrend")
    public Result<List<CountVO>> dataTrend(@RequestBody EventDataCondition searchCondition) {
        List<CountVO> dataList = new ArrayList<>();
        List<String> ids = searchCondition.getIds();
        if (CollectionUtils.isEmpty(ids)) {
            dataList = esService.groupByDate(searchCondition);
        } else {
            String[] fetchFields = new String[]{"id", "publishDate"};
            searchCondition.setFetchFields(fetchFields);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            String groupType = searchCondition.getGroupType();
            String format = null;
            if ("year".equalsIgnoreCase(groupType)) {
                format = "yyyy";
            } else if ("month".equalsIgnoreCase(groupType)) {
                format = "yyyy-MM";
            } else if ("day".equalsIgnoreCase(groupType)) {
                format = "yyyy-MM-dd";
            } else if ("hout".equalsIgnoreCase(groupType)) {
                format = "HH:00:00";
            }
            if (format != null) {
                String finalFormat = format;
                Map<String, List<SpecialInformation>> dateMap = informationList.stream()
                        .collect(Collectors .groupingBy(item -> DateUtil.formatStr(item.getPublishDate(), "yyyy-MM-dd HH:mm:ss", finalFormat)));
                for (Map.Entry<String, List<SpecialInformation>> entry : dateMap.entrySet()) {
                    CountVO countVO = new CountVO();
                    countVO.setName(entry.getKey());
                    countVO.setValue((long) entry.getValue().size());
                    dataList.add(countVO);
                }
            }
        }
        return Result.OK(dataList);
    }

    /**
     * 资讯类型分布
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    @PostMapping("/dataType/distribution")
    public Result<?> dataTypeDistribution(@RequestBody EventDataCondition searchCondition) {
        List<CountVO> dataList = new ArrayList<>();
        List<String> ids = searchCondition.getIds();
        if (CollectionUtils.isEmpty(ids)) {
            dataList = esService.groupByClassificationType(searchCondition);
        } else {
            String[] fetchFields = new String[]{"id", "classificationType"};
            searchCondition.setFetchFields(fetchFields);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            Map<Integer, List<SpecialInformation>> map = informationList.stream()
                    .collect(Collectors .groupingBy(SpecialInformation::getClassificationType));
            for (Map.Entry<Integer, List<SpecialInformation>> entry : map.entrySet()) {
                CountVO countVO = new CountVO();
                //todo  获取字典值
//              countVO.setName(entry.getKey());
                countVO.setValue((long) entry.getValue().size());
                dataList.add(countVO);
            }
        }
        return Result.OK(dataList);
    }

    /**
     * 资讯来源分析
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    @PostMapping("/dataSource")
    public Result<?> dataSource(@RequestBody EventDataCondition searchCondition){
        //todo 实现
        return Result.OK();
    }

    /**
     *资讯数量top的信息源统计
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    @PostMapping("/infoSource/dataCount/top")
    public Result<?> infoSourceDataCountTop(@RequestBody EventDataCondition searchCondition){
        List<CountVO> dataList = new ArrayList<>();
        List<String> ids = searchCondition.getIds();
        if (CollectionUtils.isEmpty(ids)) {
            dataList = esService.groupBySourceId(searchCondition);
        } else {
            String[] fetchFields = new String[]{"id", "sid"};
            searchCondition.setFetchFields(fetchFields);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            Map<String, List<SpecialInformation>> map = informationList.stream()
                    .collect(Collectors .groupingBy(SpecialInformation::getSid));
            for (Map.Entry<String, List<SpecialInformation>> entry : map.entrySet()) {
                CountVO countVO = new CountVO();
                String sid = entry.getKey();
                InfoSource infoSource = infoSourceService.getById(sid);
                countVO.setName(infoSource.getSiteName());
                countVO.setValue((long) entry.getValue().size());
                dataList.add(countVO);
            }
        }
        return Result.OK(dataList);
    }
}
