package com.zzsn.event.controller.common;

import cn.hutool.core.util.ObjectUtil;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.InfoSource;
import com.zzsn.event.entity.Subject;
import com.zzsn.event.entity.SysDictItem;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.ClbLabelService;
import com.zzsn.event.service.IInfoSourceService;
import com.zzsn.event.service.SubjectService;
import com.zzsn.event.service.SysDictItemService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.Label;
import com.zzsn.event.vo.es.SpecialInformation;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 统计分析-研究中心
 *
 * @author lkg
 * @date 2024/12/25
 */
@RestController
@RequestMapping("/statistical/analysis")
public class StatisticalAnalysisController {

    @Autowired
    private EsService esService;
    @Autowired
    private SubjectService subjectService;
    @Autowired
    private IInfoSourceService infoSourceService;
    @Autowired
    private SysDictItemService sysDictItemService;
    @Autowired
    private ClbLabelService clbLabelService;

    /**
     * 概述分析
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    @GetMapping("overview")
    public Result<OverviewAnalysisVO> overviewAnalysis(@RequestBody InfoDataSearchCondition searchCondition) {
        OverviewAnalysisVO overviewAnalysisVO = new OverviewAnalysisVO();
        Subject subject = subjectService.getById(searchCondition.getSubjectId());
        overviewAnalysisVO.setSubjectName(subject.getSubjectName());
        overviewAnalysisVO.setTimeEnable(subject.getTimeEnable());
        overviewAnalysisVO.setTimeDisable(subject.getTimeDisable());
        List<CountVO> countList = new ArrayList<>();
        int totalCount;
        List<Label> bindLabelList = bindLabelList(searchCondition.getSubjectId());
        if (CollectionUtils.isEmpty(searchCondition.getIds()) && searchCondition.getNum() == null) {
            List<String> includeValues = new ArrayList<>();
            bindLabelList.forEach(e -> includeValues.add(e.getRelationId()));
            searchCondition.setIncludeValues(includeValues.toArray(new String[0]));
            countList = esService.overView(searchCondition);
            totalCount = countList.get(0).getValue().intValue();
            countList.remove(0);
        } else {
            Map<String, Label> labelMap = bindLabelList.stream().collect(Collectors.toMap(Label::getRelationId, label -> label));
            String[] fetchFields = new String[]{"id", "sid", "labels"};
            searchCondition.setFetchFields(fetchFields);
            List<SpecialInformation> specialInformationList = esService.informationList(searchCondition);
            totalCount = specialInformationList.size();
            Map<String, List<SpecialInformation>> map = buildMap(specialInformationList, labelMap);
            for (Map.Entry<String, List<SpecialInformation>> entry : map.entrySet()) {
                CountVO countVO = new CountVO();
                String key = entry.getKey();
                countVO.setName(labelMap.get(key).getRelationName());
                List<SpecialInformation> value = entry.getValue();
                Map<String, List<SpecialInformation>> sidMap = value.stream().collect(Collectors.groupingBy(SpecialInformation::getSid));
                countVO.setValue((long) sidMap.size());
            }
        }
        overviewAnalysisVO.setInfoCount(totalCount);
        overviewAnalysisVO.setCountList(countList);
        return Result.OK(overviewAnalysisVO);
    }

    /**
     * 资讯趋势分析
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/25
     */
    @PostMapping("/dataTrend")
    public Result<List<CountVO>> dataTrend(@RequestBody InfoDataSearchCondition searchCondition) {
        List<CountVO> dataList = new ArrayList<>();
        if (CollectionUtils.isEmpty(searchCondition.getIds()) && searchCondition.getNum() == null) {
            dataList = esService.groupByDate(searchCondition);
        } else {
            String[] fetchFields = new String[]{"id", "publishDate"};
            searchCondition.setFetchFields(fetchFields);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            String groupType = searchCondition.getGroupType();
            String format = null;
            if ("year".equalsIgnoreCase(groupType)) {
                format = "yyyy";
            } else if ("month".equalsIgnoreCase(groupType)) {
                format = "yyyy-MM";
            } else if ("day".equalsIgnoreCase(groupType)) {
                format = "yyyy-MM-dd";
            } else if ("hout".equalsIgnoreCase(groupType)) {
                format = "HH:00:00";
            }
            if (format != null) {
                String finalFormat = format;
                Map<String, List<SpecialInformation>> dateMap = informationList.stream()
                        .collect(Collectors.groupingBy(item -> DateUtil.formatStr(item.getPublishDate(), "yyyy-MM-dd HH:mm:ss", finalFormat)));
                for (Map.Entry<String, List<SpecialInformation>> entry : dateMap.entrySet()) {
                    CountVO countVO = new CountVO();
                    countVO.setName(entry.getKey());
                    countVO.setValue((long) entry.getValue().size());
                    dataList.add(countVO);
                }
            }
        }
        return Result.OK(dataList);
    }

    /**
     * 资讯类型分布
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    @PostMapping("/dataType/distribution")
    public Result<?> dataTypeDistribution(@RequestBody InfoDataSearchCondition searchCondition) {
        List<CountVO> dataList = new ArrayList<>();
        if (CollectionUtils.isEmpty(searchCondition.getIds()) && searchCondition.getNum() == null) {
            dataList = esService.groupByClassificationType(searchCondition);
        } else {
            String[] fetchFields = new String[]{"id", "classificationType"};
            searchCondition.setFetchFields(fetchFields);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            Map<Integer, List<SpecialInformation>> map = informationList.stream()
                    .collect(Collectors.groupingBy(SpecialInformation::getClassificationType));
            for (Map.Entry<Integer, List<SpecialInformation>> entry : map.entrySet()) {
                CountVO countVO = new CountVO();
                countVO.setValue((long) entry.getValue().size());
                dataList.add(countVO);
            }
        }
        changeKey(dataList, "Thematic_Library");
        return Result.OK(dataList);
    }

    /**
     * 资讯来源分析
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    @PostMapping("/dataSource")
    public Result<?> dataSource(@RequestBody InfoDataSearchCondition searchCondition) {
        List<CountVO> dataList = new ArrayList<>();
        List<Label> bindLabelList = bindLabelList(searchCondition.getSubjectId());
        if (CollectionUtils.isEmpty(searchCondition.getIds()) && searchCondition.getNum() == null) {
            List<String> includeValues = new ArrayList<>();
            bindLabelList.forEach(e -> includeValues.add(e.getRelationId()));
            searchCondition.setIncludeValues(includeValues.toArray(new String[0]));
            dataList = esService.groupByInfoSourceTag(searchCondition);
        } else {
            Map<String, Label> labelMap = bindLabelList.stream().collect(Collectors.toMap(Label::getRelationId, label -> label));
            String[] fetchFields = new String[]{"id", "labels"};
            searchCondition.setFetchFields(fetchFields);
            List<SpecialInformation> specialInformationList = esService.informationList(searchCondition);
            Map<String, List<SpecialInformation>> map = buildMap(specialInformationList, labelMap);
            for (Map.Entry<String, List<SpecialInformation>> entry : map.entrySet()) {
                CountVO countVO = new CountVO();
                String key = entry.getKey();
                countVO.setName(labelMap.get(key).getRelationName());
                countVO.setValue((long) entry.getValue().size());
            }
        }
        return Result.OK(dataList);
    }

    /**
     * 资讯数量top的信息源统计
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    @PostMapping("/infoSource/dataCount/top")
    public Result<?> infoSourceDataCountTop(@RequestBody InfoDataSearchCondition searchCondition) {
        List<CountVO> dataList = new ArrayList<>();
        if (CollectionUtils.isEmpty(searchCondition.getIds()) && searchCondition.getNum() == null) {
            dataList = esService.groupBySourceId(searchCondition);
        } else {
            String[] fetchFields = new String[]{"id", "sid"};
            searchCondition.setFetchFields(fetchFields);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            Map<String, List<SpecialInformation>> map = informationList.stream()
                    .collect(Collectors.groupingBy(SpecialInformation::getSid));
            for (Map.Entry<String, List<SpecialInformation>> entry : map.entrySet()) {
                CountVO countVO = new CountVO();
                String sid = entry.getKey();
                countVO.setName(sid);
                countVO.setValue((long) entry.getValue().size());
                dataList.add(countVO);
            }
        }
        if (CollectionUtils.isNotEmpty(dataList)) {
            List<String> sidList = dataList.stream().map(CountVO::getName).collect(Collectors.toList());
            List<InfoSource> infoSources = infoSourceService.listByIds(sidList);
            Map<String, InfoSource> sourceMap = infoSources.stream().collect(Collectors.toMap(InfoSource::getId, infoSource -> infoSource));
            for (CountVO countVO : dataList) {
                String sid = countVO.getName();
                InfoSource infoSource = sourceMap.get(sid);
                countVO.setName(infoSource.getWebSiteName());
            }
        }
        return Result.OK(dataList);
    }

    /**
     * 改变key值(字典值码转为字典值)
     *
     * @param dataList 统计数据
     * @param dictCode 字典编码
     * @author lkg
     * @date 2024/12/30
     */
    private void changeKey(List<CountVO> dataList, String dictCode) {
        if (CollectionUtils.isNotEmpty(dataList)) {
            List<SysDictItem> dictItemList = sysDictItemService.listByDictCode("");
            Map<String, List<SysDictItem>> map = dictItemList.stream().collect(Collectors.groupingBy(SysDictItem::getItemValue));
            for (CountVO countVO : dataList) {
                List<SysDictItem> sysDictItemList = map.get(countVO.getName());
                if (CollectionUtils.isNotEmpty(sysDictItemList)) {
                    countVO.setName(sysDictItemList.get(0).getItemText());
                }
            }
        }
    }

    /**
     * 绑定的标签信息
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/30
     */
    private List<Label> bindLabelList(String subjectId) {
        List<Label> labels = new ArrayList<>();
        List<InfoSourceLabelVO> infoSourceLabelInfos = clbLabelService.bindInfoSourceLabelInfo(subjectId);
        if (ObjectUtil.isNotEmpty(infoSourceLabelInfos)) {
            for (InfoSourceLabelVO infoSourceLabel : infoSourceLabelInfos) {
                String labelCode = infoSourceLabel.getLabelCode();
                List<InfoSourceLabelItemVO> infoSourceLabelItemList = infoSourceLabel.getInfoSourceLabelItemList();
                for (InfoSourceLabelItemVO infoSourceLabelItem : infoSourceLabelItemList) {
                    Label label = new Label();
                    label.setRelationName(infoSourceLabelItem.getLabelItemName());
                    label.setRelationId(labelCode + "_" +infoSourceLabelItem.getLabelItemCode());
                }
            }
        }
        return labels;
    }

    /**
     * 构建map，统计数量
     *
     * @param informationList 资讯信息列表
     * @param labelMap        专题绑定的标签信息
     * @author lkg
     * @date 2025/1/2
     */
    private Map<String, List<SpecialInformation>> buildMap(List<SpecialInformation> informationList, Map<String, Label> labelMap) {
        Map<String, List<SpecialInformation>> map = new HashMap<>();
        for (SpecialInformation information : informationList) {
            List<Label> labels = information.getLabels();
            String labelId = null;
            for (Label label : labels) {
                String labelRelationId = label.getRelationId();
                Label bindLabel = labelMap.get(labelRelationId);
                if (ObjectUtil.isNotEmpty(bindLabel)) {
                    labelId = labelRelationId;
                    break;
                }
            }
            if (ObjectUtil.isNotEmpty(labelId)) {
                List<SpecialInformation> list = map.get(labelId);
                if (CollectionUtils.isEmpty(list)) {
                    list = new ArrayList<>();
                    map.put(labelId, list);
                }
                list.add(information);
            }
        }
        return map;
    }
}
