package com.zzsn.event.controller;

import com.aspose.words.Document;
import com.aspose.words.SaveFormat;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.obs.services.model.PutObjectResult;
import com.zzsn.event.config.FreeMarkerConfiguration;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.ClbFileOperationLog;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.FileDownloadCenter;
import com.zzsn.event.entity.Subject;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.*;
import com.zzsn.event.util.*;
import com.zzsn.event.util.user.AuthUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.InfoDataSearchCondition;
import com.zzsn.event.vo.LabelModelVo;
import com.zzsn.event.vo.es.AttachmentInfo;
import com.zzsn.event.vo.es.Label;
import com.zzsn.event.vo.es.SpecialInformation;
import freemarker.template.Template;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.jsoup.Jsoup;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 文件相关
 *
 * @author lkg
 * @date 2024/12/19
 */
@Slf4j
@RestController
@RequestMapping("/file")
public class FileController {

    @Autowired
    private InformationService informationService;
    @Autowired
    private IClbSysAttachmentService clbSysAttachmentService;
    @Autowired
    private ObsUtil obsUtil;
    @Autowired
    private EsService esService;
    @Autowired
    private SubjectService subjectService;
    @Autowired
    private IEventService eventService;
    @Autowired
    private FileDownloadCenterService fileDownloadCenterService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private IClbFileOperationLogService clbFileOperationLogService;

    /**
     * 专题关联信息源采集量统计-导出excel(专题列表页)
     *
     * @param subjectId 专题id/专题分类id
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @author lkg
     */
    @GetMapping("/exportStatisticsExcel")
    public void exportStatisticsExcel(@RequestParam(name = "subjectId") String subjectId,
                                      @RequestParam(name = "startDate") String startDate,
                                      @RequestParam(name = "endDate") String endDate,
                                      HttpServletResponse response) {
        try {
            List<List<String>> statistics = informationService.statisticsExportList(subjectId, startDate, endDate);
            List<String> dateList = DateUtil.betweenDate(startDate, endDate);
            List<String> headers = Stream.of("信息源id", "网站名称", "栏目名称", "栏目地址", "时间段内采集总量").collect(Collectors.toList());
            headers.addAll(dateList);
            XSSFWorkbook workbook = new XSSFWorkbook();
            ExcelExportUtil.exportExcelData(workbook, 0, headers, statistics, "信息源采集信息汇总");
            String name = "statisticsExcel.xls";
            name = new String(name.getBytes(), StandardCharsets.ISO_8859_1);
            response.setContentType("application/octet-stream;charset=ISO8859-1");
            response.setHeader("Content-Disposition", "attachment;filename=" + name);
            response.setHeader("Pargam", "no-cache");
            response.setHeader("Cache-Control", "no-cache");
            ServletOutputStream outputStream = response.getOutputStream();
            workbook.write(outputStream);
            outputStream.flush();
            outputStream.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 专题信息列表-新增页面文件上传接口
     */
    @PostMapping(value = "/upload")
    public Result<?> fileUpload(HttpServletRequest request) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        if (ObjectUtils.isEmpty(fileMap) || fileMap.size() < 1) {
            return Result.FAIL(500, "请选择文件");
        }
        List<AttachmentInfo> result = clbSysAttachmentService.fileUpload(fileMap);
        return Result.OK(result);
    }

    /**
     * 专题资讯导入模板下载
     *
     * @author lkg
     * @date 2024/06/21
     */
    @GetMapping("/downloadTemplate")
    public void downloadTemplate(HttpServletResponse response) {
        String filePath = "subjectDataImport/导入模板.xlsx";
        commonService.downloadTemplate(response, filePath);
    }

    /**
     * 批量导入数据
     *
     * @author lkg
     * @date 2024/12/25
     */
    @PostMapping("/importInfo")
    public Result<?> importInfo(HttpServletRequest request) {
        UserVo userVo = AuthUtil.getLoginUser();
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        if (fileMap.size() < 1) {
            return Result.FAIL(500, "请上传excel文件");
        }
        MultipartFile multipartFile = fileMap.get(new ArrayList<>(fileMap.keySet()).get(0));
        int index = multipartFile.getOriginalFilename().lastIndexOf(".");
        String fileSuffix = multipartFile.getOriginalFilename().substring(index + 1);
        if ("xls".equals(fileSuffix) || "xlsx".equals(fileSuffix)) {
            CompletableFuture.runAsync(()->{
                //开启日志
                ClbFileOperationLog clbFileOperationLog = clbFileOperationLogService.operationStart(multipartFile.getOriginalFilename(), userVo);
                String subjectId = request.getParameter("subjectId");
                String isTopping = request.getParameter("isTopping");
                String isExamine = request.getParameter("isExamine");
                try {
                    //1.上传文件到文件服务器
                    // 将MultipartFile的文件内容保存到字节数组
                    byte[] fileData = multipartFile.getBytes();
                    String url = null;
                    try {
                        url = obsUtil.uploadFile("SubjectUp/" + multipartFile.getOriginalFilename(), fileData).getObjectKey();
                    } catch (Exception e) {
                        clbFileOperationLogService.failure(multipartFile.getOriginalFilename(), null, "文件上传失败，无法进行解析", userVo);
                    }
                    //记录url日志
                    clbFileOperationLog.setFileUrl(url);
                    clbFileOperationLog.setSubjectId(subjectId);
                    clbFileOperationLogService.updateById(clbFileOperationLog);
                    //读取文件内容
                    List<List<String>> lists = ExcelExportUtil.readExcel(new ByteArrayInputStream(fileData), 1, 9);
                    //存入es
                    informationService.importInfo(lists, subjectId, isTopping, isExamine, clbFileOperationLog, userVo);
                } catch (Exception e) {
                    clbFileOperationLogService.failure(multipartFile.getOriginalFilename(), null, "导入异常:" + e.getMessage(), userVo);
                }
            });
            return Result.OK("正在进行处理");
        } else {
            return Result.FAIL(500, "不支持的文件类型");
        }
    }

    /**
     * 导出数据-excel格式
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/25
     */
    @PostMapping("/exportXls")
    public void exportXls(@RequestBody InfoDataSearchCondition searchCondition, HttpServletResponse response) {
        int count = esService.getCount(searchCondition);
        if (count > 0) {
            String subjectId = searchCondition.getSubjectId();
            long timestamp = System.currentTimeMillis();
            String filename = subjectId + "_" + timestamp + ".xlsx";
            String objectKey = "subjectExportFile/" + filename;
            if (count <= 1000) {
                try {
                    byte[] bytes = getBytes(searchCondition, count);
                    filename = URLEncoder.encode(filename, "UTF-8").replace("+", " ");
                    response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");
                    response.setHeader("Content-Disposition", "attachment;filename=" + filename);
                    response.setContentLength(bytes.length);
                    OutputStream out = response.getOutputStream();
                    out.write(bytes);
                    out.flush();
                    out.close();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            } else {
                //获取当前登录用户
                UserVo userVo = AuthUtil.getLoginUser();
                String subjectName;
                Integer category = searchCondition.getCategory();
                if (category == 1) {
                    Subject subject = subjectService.getById(subjectId);
                    subjectName = subject.getSubjectName();
                } else {
                    Event event = eventService.getById(subjectId);
                    subjectName = event.getEventName();
                }
                FileDownloadCenter fileDownloadCenter = new FileDownloadCenter();
                fileDownloadCenter.setRelationId(subjectId);
                String name = fileDownloadCenterService.getNoRepeatName(subjectName + "-已入库信息", userVo.getUsername());
                fileDownloadCenter.setName(name);
                fileDownloadCenter.setBuildStatus(0);
                fileDownloadCenter.setReadStatus(0);
                fileDownloadCenter.setCommitBy(userVo.getUsername());
                fileDownloadCenter.setCommitTime(new Date(timestamp));
                fileDownloadCenterService.save(fileDownloadCenter);
                String id = fileDownloadCenter.getId();
                CompletableFuture.runAsync(() -> {
                    byte[] bytes = getBytes(searchCondition, 5000);
                    LambdaUpdateWrapper<FileDownloadCenter> updateWrapper = Wrappers.lambdaUpdate();
                    updateWrapper.eq(FileDownloadCenter::getId, id);
                    updateWrapper.set(FileDownloadCenter::getCreateTime, new Date());
                    if (bytes != null) {
                        String size = FileUtils.byteCountToDisplaySize(bytes.length);
                        PutObjectResult putObjectResult = obsUtil.uploadFile(objectKey, bytes);
                        String filePath = putObjectResult.getObjectKey();
                        updateWrapper.set(FileDownloadCenter::getFilePath, filePath)
                                .set(FileDownloadCenter::getFileSize, size)
                                .set(FileDownloadCenter::getBuildStatus, 1);
                    } else {
                        updateWrapper.set(FileDownloadCenter::getBuildStatus, 2);
                    }
                    fileDownloadCenterService.update(updateWrapper);
                });
            }
        }
    }


    /**
     * 导出数据-word格式
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/25
     */
    @PostMapping("/exportDoc")
    public void exportDoc(@RequestBody InfoDataSearchCondition searchCondition, HttpServletResponse response) {
        OutputStream outputstream = null;
        ByteArrayOutputStream bos = null;
        try {
            String[] fetchFields = null;
            Integer exportType = searchCondition.getExportType();
            if (exportType == 1) {
                fetchFields = new String[]{"id", "title", "summary", "origin", "publishDate", "sourceAddress"};
            } else if (exportType == 2) {
                fetchFields = new String[]{"id", "title", "content", "origin", "publishDate", "sourceAddress"};
            }
            searchCondition.setFetchFields(fetchFields);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            Map<String, Object> map = formatDocData(informationList, null, exportType);
            Template template = FreeMarkerConfiguration.get().getTemplate("EVENT_DATA_REPORT.ftl", "UTF-8");
            bos = new ByteArrayOutputStream();
            template.process(map, new OutputStreamWriter(bos));
            Document document = new Document(new ByteArrayInputStream(bos.toByteArray()));
            bos.reset();
            document.updateFields();
            document.save(bos, SaveFormat.DOCX);
            String fileName = URLEncoder.encode("专题资讯", "UTF-8").replace("+", "-");
            response.setHeader("content-Type", "application/msword");
            response.setHeader("Content-Disposition", "attachment;filename=" + fileName);
            response.setContentLength(bos.size());
            outputstream = response.getOutputStream();
            bos.writeTo(outputstream);
            bos.close();
            outputstream.flush();
            outputstream.close();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (bos != null) {
                try {
                    bos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (outputstream != null) {
                try {
                    outputstream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }


    private byte[] getBytes(InfoDataSearchCondition searchCondition, Integer pageSize) {
        String[] fetchFields = new String[]{"score", "title", "titleRaw", "summary", "summaryRaw", "content", "contentRaw", "author", "origin", "publishDate", "sourceAddress", "classificationType", "hitWords", "labels"};
        searchCondition.setFetchFields(fetchFields);
        searchCondition.setPageSize(pageSize);
        byte[] bytes = null;
        try {
            String[] arr = new String[]{"得分", "标题", "标题译文", "摘要", "摘要译文", "正文", "正文译文", "作者", "来源", "发布时间", "网址", "专题库类型", "命中词"};
            List<String> subjectIds = new ArrayList<>();
            subjectIds.add(searchCondition.getSubjectId());
            List<LabelModelVo> labelTypeVos = commonService.subjectModelBindLabels(subjectIds);
            //动态补充表头
            List<String> headers = supplyHeaders(arr, labelTypeVos);
            SXSSFWorkbook workbook = new SXSSFWorkbook();
            for (int i = 1; ; i++) {
                searchCondition.setPageNo(i);
                List<SpecialInformation> informationList = esService.informationList(searchCondition);
                log.info("本次循环-{}，数据量为-{}", i, informationList.size());
                if (CollectionUtils.isEmpty(informationList)) {
                    break;
                }
                List<List<String>> rows = formatData(informationList, labelTypeVos);
                BigExcelExportUtil.exportExcelData(workbook, i - 1, headers, rows, "sheet" + i);
                log.info("第【{}】个sheet页写入成功", i);
            }
            // 将Workbook写入字节流
            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            workbook.write(outStream);
            // 将字节流转换为InputStream
            bytes = outStream.toByteArray();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return bytes;
    }

    private List<String> supplyHeaders(String[] arr, List<LabelModelVo> labelTypeVos) {
        List<String> headers = new ArrayList<>(Arrays.asList(arr));
        if (CollectionUtils.isNotEmpty(labelTypeVos)) {
            for (LabelModelVo labelTypeVo : labelTypeVos) {
                headers.add(labelTypeVo.getLabelName());
            }
        }
        return headers;
    }

    private List<List<String>> formatData(List<SpecialInformation> informationList, List<LabelModelVo> labelTypeVos) {
        List<List<String>> list = new ArrayList<>();
        for (SpecialInformation info : informationList) {
            List<String> data = new ArrayList<>();
            if (info.getScore() != null) {
                data.add(info.getScore() + "");
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotBlank(info.getTitle())) {
                String title = Jsoup.parse(info.getTitle()).text();
                data.add(title);
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotBlank(info.getTitleRaw())) {
                String titleRaw = Jsoup.parse(info.getTitleRaw()).text();
                data.add(titleRaw);
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotBlank(info.getSummary())) {
                String summary = Jsoup.parse(info.getSummary()).text();
                data.add(summary);
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotBlank(info.getSummaryRaw())) {
                String summaryRaw = Jsoup.parse(info.getSummaryRaw()).text();
                data.add(summaryRaw);
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotBlank(info.getContent())) {
                String content = Jsoup.parse(info.getContent()).text();
                data.add(content);
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotBlank(info.getContentRaw())) {
                String contentRaw = Jsoup.parse(info.getContentRaw()).text();
                data.add(contentRaw);
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotBlank(info.getAuthor())) {
                String author = Jsoup.parse(info.getAuthor()).text();
                data.add(author);
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotEmpty(info.getOrigin())) {
                data.add(info.getOrigin());
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotEmpty(info.getPublishDate())) {
                data.add(info.getPublishDate());
            } else {
                data.add(" ");
            }
            if (StringUtils.isNotEmpty(info.getSourceAddress())) {
                data.add(info.getSourceAddress());
            } else {
                data.add(" ");
            }
            if (info.getClassificationType() != null) {
                if (info.getClassificationType() == 1) {
                    data.add("政策");
                } else if (info.getClassificationType() == 2) {
                    data.add("领导讲话");
                } else if (info.getClassificationType() == 3) {
                    data.add("专家观点");
                } else if (info.getClassificationType() == 4) {
                    data.add("企业案例");
                } else {
                    data.add("其它");
                }
            } else {
                data.add(" ");
            }
            if (CollectionUtils.isNotEmpty(info.getHitWords())) {
                data.add(StringUtils.join(info.getHitWords(), ","));
            } else {
                data.add(" ");
            }
            //标签处理
            if (CollectionUtils.isNotEmpty(labelTypeVos)) {
                List<Label> relationLabels = info.getLabels();
                Map<String, List<Label>> map = new HashMap<>(labelTypeVos.size());
                for (LabelModelVo labelTypeVo : labelTypeVos) {
                    String labelMark = labelTypeVo.getLabelMark();
                    Integer orderNo = labelTypeVo.getOrderNo();
                    String key = labelMark + "&&" + orderNo;
                    List<Label> labels = new ArrayList<>();
                    if (CollectionUtils.isNotEmpty(relationLabels)) {
                        labels = relationLabels.stream()
                                .filter(e -> StringUtils.isNotEmpty(e.getLabelMark()) &&
                                        (e.getLabelMark().equals(labelMark) || e.getLabelMark().contains(labelMark)))
                                .collect(Collectors.toList());
                    }
                    List<Label> labelList = map.get(key);
                    if (CollectionUtils.isEmpty(labelList)) {
                        map.put(key, labels);
                    } else {
                        labelList.addAll(labels);
                        map.put(key, labelList);
                    }
                }
                for (LabelModelVo labelTypeVo : labelTypeVos) {
                    String labelMark = labelTypeVo.getLabelMark();
                    Integer orderNo = labelTypeVo.getOrderNo();
                    String key = labelMark + "&&" + orderNo;
                    List<Label> labelList = map.get(key);
                    if (CollectionUtils.isEmpty(labelList)) {
                        List<Label> labels = new ArrayList<>();
                        map.put(key, labels);
                    }
                }
                //排序，和表头对应
                LinkedHashMap<String, List<Label>> sortMap = map.entrySet().stream()
                        .sorted((o1, o2) -> {
                            Integer key1 = Integer.parseInt(o1.getKey().split("&&")[1]);
                            Integer key2 = Integer.parseInt(o2.getKey().split("&&")[1]);
                            return key1.compareTo(key2);
                        })
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue,
                                (oldValue, newValue) -> oldValue, LinkedHashMap::new));
                for (Map.Entry<String, List<Label>> entry : sortMap.entrySet()) {
                    List<Label> value = entry.getValue();
                    if (CollectionUtils.isEmpty(value)) {
                        data.add(" ");
                    } else {
                        StringBuilder sb = new StringBuilder();
                        for (Label label : value) {
                            sb.append(",").append(label.getRelationName());
                        }
                        data.add(sb.substring(1));
                    }
                }
            }
            list.add(data);
        }
        return list;
    }

    /**
     * 格式化ftl文件所需要的数据格式
     *
     * @param exportDataList 资讯列表
     * @param searchWord     搜索词
     * @param type           导出方式(1-摘要;2-正文)
     * @author lkg
     * @date 2024/4/11
     */
    private Map<String, Object> formatDocData(List<SpecialInformation> exportDataList, String searchWord, Integer type) {
        String docTitle = "事件资讯";
        //遍历取到的文章
        List<Map<String, Object>> contents = new ArrayList<>();
        //文档结构图
        Map<String, List<Map<String, Object>>> contentSortMap = new LinkedHashMap<>();//文档结构图
        //目录
        Map<String, List<Map<String, Object>>> catalogSortMap = new LinkedHashMap<String, List<Map<String, Object>>>();//目录
        List<Map<String, Object>> caList = new ArrayList<Map<String, Object>>();
        catalogSortMap.put(docTitle, caList);//目录
        int num = 1;
        for (SpecialInformation subjectDataVo : exportDataList) {
            String rid = subjectDataVo.getId();
            String title = Utility.getValueAfterReplaceSpecialWordNotEnter(subjectDataVo.getTitle());
            String sourceAddress = subjectDataVo.getSourceAddress();
            String origin = subjectDataVo.getOrigin();
            String publishDate = subjectDataVo.getPublishDate();
            String info = StringUtils.isNotEmpty(origin) ? "（来源：" + origin + "，发布时间：" + publishDate + "）" : "（发布时间：" + publishDate + "）";
            Map<String, Object> content = new HashMap<>();
            content.put("id", rid);
            content.put("title", title);
            content.put("url", Utility.getValueAfterReplaceSpecialWordNotEnter(sourceAddress));
            if (type == 1) {
                String summary = subjectDataVo.getSummary();
                if (StringUtils.isNotEmpty(summary)) {
                    summary = Utility.getValueAfterReplaceSpecialWordNotEnter(Utility.TransferHTML2Text(summary));
                }
                content.put("info", summary + info);
            } else {
                content.put("info", info);
                String contentStr = subjectDataVo.getContent();
                if (StringUtils.isNotEmpty(contentStr)) {
                    contentStr = Utility.getValueAfterReplaceSpecialWordNotEnter(Utility.TransferHTML2Text(contentStr));
                }
                content.put("contentStr", contentStr);
            }
            contents.add(content);
            contentSortMap.put(docTitle, contents);//文档结构图中，文章标题
            Map<String, Object> catalog = new HashMap<>();
            catalog.put("index", num);
            catalog.put("title", title);
            catalog.put("id", rid + "-" + (num - 1));
            caList.add(catalog);
            catalogSortMap.put(docTitle, caList);
            num++;
        }
        Map<String, Object> map = new HashMap<>();
        map.put("docTitle", docTitle);
        map.put("publishDate", DateUtil.dateToString(new Date(), "yyyy-MM-dd"));
        map.put("keyWords", StringUtils.isEmpty(searchWord) ? "" : searchWord);
        map.put("catalogMap", catalogSortMap);
        map.put("contentMap", contentSortMap);
        return map;
    }
}
