package com.zzsn.event.task;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONWriter;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.AnalysisService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.SubjectAnalysisService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.vo.KafkaDataVo;
import com.zzsn.event.vo.PropagationPathVo;
import com.zzsn.event.vo.SubjectDataVo;
import com.zzsn.event.vo.SubjectKafkaVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CompletableFuture;

/**
 * @author lkg
 * @description: 专题分析定时任务
 * @date 2022/7/8 11:58
 */
@Slf4j
@Component
public class AnalysisTask {

    @Autowired
    private IEventService eventService;
    @Autowired
    private EsService esService;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private AnalysisService analysisService;
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;

    @Value("${scheduling.yjzxEnable:false}")
    Boolean yjzxEnable;
    /**
     * 每天凌晨0点20分执行一次
     * 发送 伪事件脉络 所需信息到kafka对应的topic
     */
    @Scheduled(cron = "0 20 0 * * ?")
    public void eventContext_fake() {
        if(yjzxEnable){
            //研究中心不需要此任务
            return;
        }
        Date today = new Date();
        Date disableDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> fakeEventIdList = new ArrayList<>();
        List<SubjectKafkaVo> subjectList = eventService.progressList(disableDate);
        subjectList.forEach(e -> {
            LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectAnalysis::getSubjectId, e.getId()).eq(SubjectAnalysis::getCategory, 2);
            int count = subjectAnalysisService.count(queryWrapper);
            if (count < Constants.FAKE_NUM) {
                fakeEventIdList.add(e);
            }
        });
        for (SubjectKafkaVo subjectKafkaVo : fakeEventIdList) {
            List<KafkaDataVo> kafkaDataVoList = new ArrayList<>();
            String subjectId = subjectKafkaVo.getId();
            String keyWord = subjectKafkaVo.getKeyWord();
            List<SubjectDataVo> dataList = esService.getDataBySubjectId(subjectId, null, null,  Constants.FETCH_FIELDS_DATA);
            format(subjectId, kafkaDataVoList, dataList);
            splitSend(Constants.FAKE_EVENT_CONTEXT_SEND_TOPIC, subjectId, kafkaDataVoList, keyWord);
        }
    }

    private void format(String subjectId, List<KafkaDataVo> kafkaDataVoList, List<SubjectDataVo> dataList) {
        dataList.forEach(e -> {
            String dataId = e.getId();
            KafkaDataVo kafkaDataVo = new KafkaDataVo();
            BeanUtils.copyProperties(e, kafkaDataVo);
            List<String> idList = new ArrayList<>();
            idList.add(dataId);
            Map<String, Integer> similarNumber = esService.getSimilarNumber(subjectId, idList);
            Integer count = similarNumber.get(dataId);
            if (count == null) {
                count = 0;
            }
            kafkaDataVo.setRepeatNum(count);
            kafkaDataVoList.add(kafkaDataVo);
        });
    }

    //防止数据量太大，超过kafka的最大值，所以分批发送，一次20条数据
    private void splitSend(String topic, String subjectId, List<KafkaDataVo> list, String keyWord) {
        List<List<KafkaDataVo>> partition = ListUtils.partition(list, 20);
        partition.forEach(e -> {
            Map<String, Object> map = new HashMap<>();
            map.put("keyword", keyWord);
            map.put("data", e);
            kafkaTemplate.send(topic, subjectId, JSON.toJSONString(map, JSONWriter.Feature.WriteMapNullValue));
        });
    }

    /**
     * 定时生成传播路径
     * 每天凌晨0点10分执行一次
     */
    @Scheduled(cron = "0 0 1 * * ?")
    public void propagationPath() {
        Date today = new Date();
        Date deadlineDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> subjects = eventService.eventSubjectList();
        for (SubjectKafkaVo subject : subjects) {
            CompletableFuture.runAsync(()->{
                String subjectId = subject.getId();
                int count = esService.count(subjectId, null, null);
                if (count > 0) {
                    String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.PROPAGATION_KEY + subjectId;
                    Date timeDisable = subject.getTimeDisable();
                    //已经结束的事件专题，永久缓存
                    if (timeDisable != null && deadlineDate.compareTo(timeDisable) > 0) {
                        Object cacheObject = redisUtil.get(key);
                        if (cacheObject == null) {
                            PropagationPathVo pathVo = analysisService.propagationPath(subjectId);
                            if (ObjectUtils.isNotEmpty(pathVo)) {
                                redisUtil.set(key, pathVo);
                                log.info("专题-{},传播路径数据【永久】缓存成功!", subject.getSubjectName());
                            }
                        }
                    } else {//已经结束的事件专题，缓存有效期一天
                        PropagationPathVo pathVo = analysisService.propagationPath(subjectId);
                        if (ObjectUtils.isNotEmpty(pathVo)) {
                            redisUtil.set(key, pathVo, 3600 * 24);
                            log.info("专题-{},传播路径数据缓存成功!", subject.getSubjectName());
                        }
                    }
                }
            });
        }
    }
}
