package com.zzsn.event.controller;

import com.aspose.words.Document;
import com.aspose.words.SaveFormat;
import com.zzsn.event.config.FreeMarkerConfiguration;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.ExcelExportUtil;
import com.zzsn.event.util.Utility;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.EventExcelVO;
import com.zzsn.event.vo.ExportParam;
import com.zzsn.event.vo.SubjectDataVo;
import freemarker.template.Template;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.util.*;

/**
 * 导出功能
 *
 * @author lkg
 * @date 2024/4/10
 */
@RestController
@RequestMapping("/export")
public class EventExportController {

    @Autowired
    private IEventService eventService;
    @Autowired
    private EsService esService;

    /**
     * 导出事件列表
     *
     * @param exportParam 导出参数封装
     * @author lkg
     * @date 2024/4/10
     */
    @PostMapping("/eventList")
    public void exportEventList(@RequestBody ExportParam exportParam, HttpServletResponse response) {
        UserVo currentUser = UserUtil.getLoginUser();
        String username = currentUser.getUsername();
        String[] headers = new String[]{"id", "事件名称", "事件描述", "发布时间", "热度"};
        List<EventExcelVO> eventList = getEventList(exportParam,username);
        if (CollectionUtils.isNotEmpty(eventList)) {
            String name = "event.xlsx";
            List<List<String>> dataList = new ArrayList<>();
            eventList.forEach(e -> dataList.add(e.toList()));
            XSSFWorkbook workbook = ExcelExportUtil.exportExcelData(Arrays.asList(headers), dataList, "事件");
            try {
                setResponseHeader(response, name);
                ServletOutputStream outputStream = response.getOutputStream();
                workbook.write(outputStream);
                outputStream.flush();
                outputStream.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 导出事件资讯列表
     *
     * @param exportParam 参数封装
     * @author lkg
     * @date 2024/4/11
     */
    @PostMapping("/dataList")
    public void exportDataList(@RequestBody ExportParam exportParam, HttpServletResponse response) {
        UserVo currentUser = UserUtil.getLoginUser();
        String username = currentUser.getUsername();
        OutputStream outputstream = null;
        ByteArrayOutputStream bos = null;
        try {
            List<String> eventIdList = exportParam.getEventIdList();
            if (CollectionUtils.isEmpty(eventIdList)) {
                List<EventExcelVO> frontList = getEventList(exportParam,username);
                frontList.forEach(e -> eventIdList.add(e.getId()));
            }
            List<SubjectDataVo> exportDataList = esService.exportDataList(eventIdList, exportParam.getSearchWord(), exportParam.getPosition(), exportParam.getCategory(),
                    exportParam.getArticleIdList(), exportParam.getColumn(), exportParam.getOrder(), exportParam.getType(), exportParam.getSize());
            Map<String, Object> map = formatDocData(exportDataList, exportParam.getSearchWord(), exportParam.getType());
            Template template = FreeMarkerConfiguration.get().getTemplate("EVENT_DATA_REPORT.ftl", "UTF-8");
            bos = new ByteArrayOutputStream();
            template.process(map, new OutputStreamWriter(bos));
            Document document = new Document(new ByteArrayInputStream(bos.toByteArray()));
            bos.reset();
            document.updateFields();
            document.save(bos, SaveFormat.DOCX);
            String fileName = URLEncoder.encode("事件资讯", "UTF-8").replace("+", "-");
            response.setHeader("content-Type", "application/msword");
            response.setHeader("Content-Disposition", "attachment;filename=" + fileName);
            response.setContentLength(bos.size());
            outputstream = response.getOutputStream();
            bos.writeTo(outputstream);
            bos.close();
            outputstream.flush();
            outputstream.close();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (bos != null) {
                try {
                    bos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (outputstream != null) {
                try {
                    outputstream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }


    private List<EventExcelVO> getEventList(ExportParam exportParam,String createBy){
        List<EventExcelVO> eventList = new ArrayList<>();
        Integer category = exportParam.getClassify();
        if (category == 1) {
            eventList = eventService.frontAllList(exportParam.getProjectId(),createBy,exportParam.getEventIdList(),exportParam.getEventName()
            ,exportParam.getEventType(),exportParam.getLabelField(),exportParam.getLabelName(),exportParam.getSize());
        } else if (category == 3) {
            eventList = eventService.frontOwnerList(exportParam.getProjectId(),createBy,exportParam.getEventIdList(),exportParam.getEventName()
                    ,exportParam.getEventType(),exportParam.getLabelField(),exportParam.getLabelName(),exportParam.getSize());
        }
        return eventList;
    }

    /**
     * 格式化ftl文件所需要的数据格式
     *
     * @param exportDataList 资讯列表
     * @param searchWord     搜索词
     * @param type           导出方式(1-摘要;2-正文)
     * @author lkg
     * @date 2024/4/11
     */
    private Map<String, Object> formatDocData(List<SubjectDataVo> exportDataList, String searchWord, Integer type) {
        String docTitle = "事件资讯";
        //遍历取到的文章
        List<Map<String, Object>> contents = new ArrayList<>();
        //文档结构图
        Map<String, List<Map<String, Object>>> contentSortMap = new LinkedHashMap<>();//文档结构图
        //目录
        Map<String, List<Map<String, Object>>> catalogSortMap = new LinkedHashMap<String, List<Map<String, Object>>>();//目录
        List<Map<String, Object>> caList = new ArrayList<Map<String, Object>>();
        catalogSortMap.put(docTitle, caList);//目录
        int num = 1;
        for (SubjectDataVo subjectDataVo : exportDataList) {
            String rid = subjectDataVo.getId();
            String title = Utility.getValueAfterReplaceSpecialWordNotEnter(subjectDataVo.getTitle());
            String sourceAddress = subjectDataVo.getSourceAddress();
            String origin = subjectDataVo.getOrigin();
            String summary = subjectDataVo.getSummary();
            if (StringUtils.isNotEmpty(summary)) {
                summary = Utility.getValueAfterReplaceSpecialWordNotEnter(Utility.TransferHTML2Text(summary));
            }
            String publishDate = subjectDataVo.getPublishDate();
            String contentStr = subjectDataVo.getContent();
            if (StringUtils.isNotEmpty(contentStr)) {
                contentStr = Utility.getValueAfterReplaceSpecialWordNotEnter(Utility.TransferHTML2Text(contentStr));
            }
            String info = StringUtils.isNotEmpty(origin) ? "（来源：" + origin + "，发布时间：" + publishDate + "）" : "（发布时间：" + publishDate + "）";
            Map<String, Object> content = new HashMap<>();
            content.put("id", rid);
            content.put("title", title);
            content.put("url", Utility.getValueAfterReplaceSpecialWordNotEnter(sourceAddress));
            if (type == 1) {
                content.put("info", summary + info);
            } else {
                content.put("info", info);
                content.put("contentStr", contentStr);
            }
            contents.add(content);
            contentSortMap.put(docTitle, contents);//文档结构图中，文章标题
            Map<String, Object> catalog = new HashMap<>();
            catalog.put("index", num);
            catalog.put("title", title);
            catalog.put("id", rid + "-" + (num - 1));
            caList.add(catalog);
            catalogSortMap.put(docTitle, caList);
            num++;
        }
        Map<String, Object> map = new HashMap<>();
        map.put("docTitle", docTitle);
        map.put("publishDate", DateUtil.dateToString(new Date(), "yyyy-MM-dd"));
        map.put("keyWords", StringUtils.isEmpty(searchWord) ? "" : searchWord);
        map.put("catalogMap", catalogSortMap);
        map.put("contentMap", contentSortMap);
        return map;
    }

    private void setResponseHeader(HttpServletResponse response, String name) {
        try {
            try {
                name = new String(name.getBytes(), "ISO8859-1");
            } catch (Exception e) {
                e.printStackTrace();
            }
            response.setContentType("application/octet-stream;charset=ISO8859-1");
            response.setHeader("Content-Disposition", "attachment;filename=" + name);
            response.setHeader("Pargam", "no-cache");
            response.setHeader("Cache-Control", "no-cache");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
