package com.zzsn.event.controller;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.clb.common.model.task.dto.titr.KeyWordsDTO;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.KeyWords;
import com.zzsn.event.entity.SubjectInfoSourceMap;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.IKeyWordsService;
import com.zzsn.event.service.ISubjectInfoSourceMapService;
import com.zzsn.event.service.LabelEntityService;
import com.zzsn.event.util.HttpUtil;
import com.zzsn.event.util.ObjectUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CompletableFuture;

/**
 * 事件后台管理
 *
 * @author lkg
 * @date 2024/4/8
 */
@Slf4j
@Api(tags = "事件后台管理")
@RestController
@RequestMapping("/manage")
public class EventManageController {

    @Autowired
    private IEventService eventService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;

    @Autowired
    private IKeyWordsService keyWordsService;

    @Autowired
    private RedisUtil redisUtil;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;

    @Value(("${serviceProject.url:}"))
    private String SERVICE_PROJECT_URL;
    @Value("${kafka.topic.event.run:}")
    private String EVENT_MODEL_KAFKA_CHANNEL;

    /**
     * 分页列表-后台管理
     *
     * @param eventName     事件名称
     * @param eventType     事件分类id
     * @param facePublic    是否公开(0-未公开;1-已公开)
     * @param publishStatus 发布状态(0-未发布;1-已发布)
     * @param startTime     开始时间
     * @param endTime       结束时间
     * @param order         排序字段
     * @param orderType     排序方式
     * @param pageNo        偏移量
     * @param pageSize      返回条数
     * @author lkg
     * @date 2024/4/8
     */
    @ApiOperation(value = "事件-分页列表查询", notes = "事件-分页列表查询")
    @GetMapping(value = "/pageList")
    public Result<?> pageList(@RequestParam(name = "eventName", required = false) String eventName,
                              @RequestParam(name = "eventType", required = false) Integer eventType,
                              @RequestParam(name = "facePublic", required = false) Integer facePublic,
                              @RequestParam(name = "publishStatus", required = false) Integer publishStatus,
                              @RequestParam(name = "startTime", required = false) String startTime,
                              @RequestParam(name = "endTime", required = false) String endTime,
                              @RequestParam(name = "order", required = false) String order,
                              @RequestParam(name = "orderType", defaultValue = "asc") String orderType,
                              @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<EventManageVO> pageList = eventService.pageList(eventName, eventType, facePublic, publishStatus, startTime, endTime, order, orderType, pageNo, pageSize);
        return Result.OK(pageList);
    }


    /**
     * 地域信息-树型结构
     *
     * @param type 类别(1-国际;2-国内)
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping("/regionTree")
    public Result<?> regionTree(@RequestParam Integer type) {
        List<LabelTypeVO> nodes = labelEntityService.regionTree(type);
        return Result.OK(nodes);
    }

    /**
     * 通过id查询
     *
     * @param id 事件id
     * @return
     */
    @GetMapping(value = "/queryById")
    public Result<?> queryById(@RequestParam(name = "id") String id) {
        EventVO eventVO = eventService.queryInfo(id);
        return Result.OK(eventVO);
    }



    /**
     * 1.2 添加
     *
     * @param eventParam
     * @return
     */
    @PostMapping(value = "/add")
    public Result<?> add(@RequestBody AddEventParam eventParam) {
        KeywordsVO keywordsVO = eventParam.getKeywordsVO();
        if (keywordsVO != null) {
            UserVo currentUser = UserUtil.getLoginUser();
            eventParam.setCreateBy(currentUser.getUsername());
            Event event = eventService.saveMain(eventParam);
            //新增关键词
            KeyWords keyWords = keyWordsService.saveKeyword(event, keywordsVO.getKeyword(), keywordsVO.getExclusionWord());
            CompletableFuture.runAsync(() -> {
                //插入xxlJob
                iXxlJobInfoService.eventInsert(event);
                //关键词入缓存
                KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                BeanUtils.copyProperties(keyWords, redisKeywordDTO);
                redisKeywordDTO.setStartTime(event.getStartTime());
                redisKeywordDTO.setEndTime(event.getEndTime());
                redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWords.getWordsCode(), redisKeywordDTO);
                //插入xxljob
                iXxlJobInfoService.keyWordsInsert(redisKeywordDTO);
                //为了立即响应，关键词新增时放入消息队列
                kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
                kafkaTemplate.send(EVENT_MODEL_KAFKA_CHANNEL, event.getEventCode());

            });
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 1.3 编辑
     *
     * @param addEventParam
     * @return
     */
    @PostMapping(value = "/edit")
    public Result<?> edit(@RequestBody AddEventParam addEventParam) {
        //关键词
        KeywordsVO keywordsVO = addEventParam.getKeywordsVO();
        if (keywordsVO != null) {
            UserVo currentUser = UserUtil.getLoginUser();
            addEventParam.setUpdateTime(new Date());
            addEventParam.setUpdateBy(currentUser.getUsername());
            eventService.updateMain(addEventParam);
            keyWordsService.update(Wrappers.<KeyWords>lambdaUpdate().eq(KeyWords::getId, keywordsVO.getId())
                    .set(KeyWords::getKeyWord, keywordsVO.getKeyword())
                    .set(KeyWords::getExclusionWord, keywordsVO.getExclusionWord()));
            CompletableFuture.runAsync(() -> {
                KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                BeanUtils.copyProperties(keywordsVO, redisKeywordDTO);
                redisKeywordDTO.setKeyWord(keywordsVO.getKeyword());
                redisKeywordDTO.setStartTime(addEventParam.getStartTime());
                redisKeywordDTO.setEndTime(addEventParam.getEndTime());
                redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keywordsVO.getWordsCode(), redisKeywordDTO);
                //为了立即响应，关键词编辑时放入消息队列
                kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
            });
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 启用
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/9/12
     */
    @GetMapping("/enable")
    public Result<?> enable(@RequestParam String eventId) {
        eventService.updateStatus(eventId, 1);
        return Result.OK();
    }

    /**
     * 启用
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/9/12
     */
    @GetMapping("/disable")
    public Result<?> disable(@RequestParam String eventId) {
        eventService.updateStatus(eventId, 0);
        return Result.OK();
    }


    /**
     * 通过id删除
     *
     * @param id 事件id
     * @return
     */
    @GetMapping(value = "/delete")
    public Result<?> delete(@RequestParam(name = "id") String id) {
        eventService.deleteMain(id);
        CompletableFuture.runAsync(() -> {
            EventVO eventVO = eventService.queryInfo(id);
            iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
            //删除关键词
            KeywordsVO keywordsVO = eventVO.getKeywordsVO();
            if (keywordsVO != null) {
                String wordsCode = keywordsVO.getWordsCode();
                keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + wordsCode);
                iXxlJobInfoService.deleteByInfosourceCode(wordsCode);
                //todo 是否删除对应的资讯数据
            }
        });
        return Result.OK();
    }

    /**
     * 批量删除
     *
     * @param ids 事件id，多个用逗号隔开
     * @author lkg
     * @date 2024/12/17
     */
    @GetMapping(value = "/deleteBatch")
    public Result<?> deleteBatch(@RequestParam(name = "ids") String ids) {
        String[] idArr = ids.split(",");
        //删除
        for (String id : idArr) {
            eventService.deleteMain(id);
        }
        //异步删除关联关系
        CompletableFuture.runAsync(() -> {
            for (String id : idArr) {
                EventVO eventVO = eventService.queryInfo(id);
                iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
                //删除关键词
                KeywordsVO keywordsVO = eventVO.getKeywordsVO();
                if (keywordsVO != null) {
                    String wordsCode = keywordsVO.getWordsCode();
                    keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                    redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + wordsCode);
                    iXxlJobInfoService.deleteByInfosourceCode(wordsCode);
                }
            }
        });
        //todo 是否删除对应的资讯数据
        return Result.OK();
    }


    /**
     * 1.7 发布
     *
     * @return
     */
    @PostMapping(value = "/publish")
    public Result<?> publish(@RequestBody Event event) {
        Event byId = eventService.getById(event.getId());
        byId.setUpdateTime(new Date());
        byId.setPublishDate(event.getPublishDate());
        byId.setPublishStatus(event.getPublishStatus());
        eventService.updateById(byId);
        return Result.OK();
    }



    /**
     * @param event 条件封装
     * @描述 查询事件列表
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/listAll")
    public Result<?> listAll(@RequestBody Event event) {
        LambdaQueryWrapper<Event> queryWrapper = Wrappers.lambdaQuery();
        setListAllQueryWrapper(event, queryWrapper);
        List<Event> list = eventService.list(queryWrapper);
        return Result.OK(list);
    }

    /**
     * @param event    条件封装
     * @param pageNo
     * @param pageSize
     * @描述 查询事件列表（分页，产业链关联事件时查询用）
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/pageListAll")
    public Result<?> pageListAll(@RequestBody Event event, @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        LambdaQueryWrapper<Event> queryWrapper = Wrappers.lambdaQuery();
        setListAllQueryWrapper(event, queryWrapper);
        Page<Event> page = new Page<>(pageNo, pageSize);
        IPage<Event> pageList = eventService.page(page, queryWrapper);
        return Result.OK(pageList);
    }
    private void setListAllQueryWrapper(@RequestBody Event event, LambdaQueryWrapper<Event> queryWrapper) {
        String eventType = event.getEventType();
        String startDate = event.getStartDate();
        String endDate = event.getEndDate();
        String keyword = event.getKeyword();
        queryWrapper.eq(StringUtils.isNotEmpty(eventType), Event::getEventType, eventType);
        if (StringUtils.isNotEmpty(startDate)) {
            queryWrapper.ge(Event::getCreateTime, startDate);
        }
        if (StringUtils.isNotEmpty(endDate)) {
            queryWrapper.le(Event::getCreateTime, endDate);
        }
        if (StringUtils.isNotEmpty(keyword)) {
            queryWrapper.and(q -> {
                q.eq(Event::getEventType, keyword).or().like(Event::getEventName, keyword).or().eq(Event::getEventDescribe, keyword);
            });
        }
    }

    /**
     * 2.1 专题信息源绑定
     */
    @PostMapping("/infoSourceBind")
    public Object infoSourceBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/infoSource/infoSourceBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.2 查询信息源组的绑定列表
     */
    @GetMapping("/bindList")
    public Object bindList(InfoSourceVo infoSourceVo,
                           @RequestParam(name = "ynBind") Integer ynBind,
                           @RequestParam(name = "groupId") String groupId,
                           @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                           @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        String url = SERVICE_PROJECT_URL + "event/infoSource/bindList";
        Map<String, String> params = ObjectUtil.objectToMap(infoSourceVo);
        params.put("ynBind", ynBind.toString());
        params.put("groupId", groupId);
        params.put("pageNo", pageNo.toString());
        params.put("pageSize", pageSize.toString());
        return HttpUtil.doGet(url, params, null,"utf-8");
    }

    /**
     * 2.3 关键词类别
     */
    @GetMapping("/keywordsType/rootListNoPage")
    public Object keywordsList(@RequestParam(name = "contain", defaultValue = "false") Boolean contain,
                               @RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {
        String url = SERVICE_PROJECT_URL + "event/keywordsType/rootListNoPage";
        Map<String, String> params = new HashMap<>();
        params.put("contain", contain.toString());
        params.put("subjectId", subjectId);
        return HttpUtil.doGet(url, params, null,"utf-8");
    }

    /**
     * 2.4 专题关键词绑定
     */
    @PostMapping("/keyWordsBind")
    public Object keyWordsBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/keyWordsBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.5 专题关键词绑定
     */
    @PostMapping("/keyWordsEdit")
    public Object keyWordsEdit(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectKeywordsMap);
            String url = SERVICE_PROJECT_URL + "event/keyWords/edit";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.6 标签查询
     */
    @PostMapping("/label/treeList")
    public Object labelTreeList(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectKeywordsMap);
            String url = SERVICE_PROJECT_URL + "event/label/treeList";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.7 专题绑定的搜索引擎列表
     *
     * @return
     */
    @ApiOperation(value = "专题绑定的搜索引擎列表", notes = "专题绑定的搜索引擎列表")
    @GetMapping(value = "/bindSearchEngineList")
    public Object bindSearchEngineList(SearchEnginesVo searchEnginesVo) {
        String url = SERVICE_PROJECT_URL + "event/bindSearchEngineList";
        Map<String, String> params = new HashMap<>();
        params.put("searchEnginesVo", searchEnginesVo.toString());
        params.put("subjectId", searchEnginesVo.getSubjectId());
        if (searchEnginesVo.getType() != null) {
            params.put("type", searchEnginesVo.getType().toString());
        }
        return HttpUtil.doGet(url, params, null,"utf-8");
    }

    /**
     * 2.8 专题搜索引擎绑定
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "专题搜索引擎绑定", notes = "专题搜索引擎绑定")
    @PostMapping(value = "/searchEngineBind")
    public Object searchEngineBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/searchEngineBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.9 专题绑定的信息源组列表
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "专题绑定/排除的信息源组列表", notes = "专题绑定/排除的信息源组列表")
    @GetMapping(value = "/bindInfoSourceList")
    public Object bindInfoSourceList(@RequestParam(name = "id") String id) {
        String url = SERVICE_PROJECT_URL + "event/bindInfoSourceList";
        Map<String, String> params = new HashMap<>();
        params.put("id", id);
        return HttpUtil.doGet(url, params, null,"utf-8");
    }

    /**
     * 2.10 专题信息源绑定删除
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "专题信息源绑定删除", notes = "专题信息源绑定删除")
    @PostMapping(value = "/deleteBindInfoSource")
    public Object deleteBindInfoSource(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            ;
            String url = SERVICE_PROJECT_URL + "event/deleteBindInfoSource";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.11 专题与信息源关联编辑
     *
     * @param subjectInfoSourceMap
     * @return
     */

    @ApiOperation(value = "专题与信息源关联表-编辑", notes = "专题与信息源关联表-编辑")
    @PostMapping(value = "/SubjectInfoSourceMap/edit")
    public Result<?> edit(@RequestBody SubjectInfoSourceMap subjectInfoSourceMap) {
        subjectInfoSourceMapService.updateMain(subjectInfoSourceMap);
        return Result.OK("编辑成功!");
    }

    /**
     * 2.12 信息源组类别列表查询
     *
     * @param contain
     * @param subjectId
     * @return
     */
    @ApiOperation(value = "信息源组类别-列表查询", notes = "信息源组类别-列表查询")
    @GetMapping(value = "/groupType/rootListNoPage")
    public Object queryPageListNoPage(@RequestParam(name = "contain", defaultValue = "false") Boolean contain,
                                      @RequestParam(name = "keywords", required = false) String keywords,
                                      @RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {
        String url = SERVICE_PROJECT_URL + "event/groupType/rootListNoPage";
        Map<String, String> params = new HashMap<>();
        params.put("contain", contain.toString());
        params.put("keywords", keywords);
        params.put("subjectId", subjectId);
        return HttpUtil.doGet(url, params, null,"utf-8");
    }

    /**
     * 2.13 专题绑定的关键词组列表
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "专题绑定的关键词组列表", notes = "专题绑定的关键词组列表")
    @GetMapping(value = "/bindKeyWordsList")
    public Object bindKeyWordsList(@RequestParam(name = "id") String id) {
        String url = SERVICE_PROJECT_URL + "event/bindKeyWordsList";
        Map<String, String> params = new HashMap<>();
        params.put("id", id);
        return HttpUtil.doGet(url, params, null,"utf-8");
    }

    /**
     * 2.14 删除专题关键词绑定
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "删除专题关键词绑定", notes = "删除专题关键词绑定")
    @PostMapping(value = "/deleteKeyWordsBind")
    public Object deleteKeyWordsBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/deleteKeyWordsBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.15 关键词管理-分页列表查询
     *
     * @param keyWords
     * @param pageNo
     * @param pageSize
     * @return
     */
    @ApiOperation(value = "关键词管理-分页列表查询", notes = "关键词管理-分页列表查询")
    @GetMapping(value = "/keyWords/listByTypeId")
    public Object queryPageListByTypeId(KeyWords keyWords,
                                        @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                        @RequestParam(name = "wordsName", required = false) String wordsName,
                                        @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                        @RequestParam(name = "keyWordsTypeId", defaultValue = "0") String keyWordsTypeId,
                                        @RequestParam(name = "search", defaultValue = "false") Boolean search,
                                        @RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {

        String url = SERVICE_PROJECT_URL + "event/keyWords/listByTypeId";
        Map<String, String> params = new HashMap<>();
        keyWords.setWordsName(wordsName);
        params.put("wordsName", wordsName);
        params.put("pageNo", pageNo.toString());
        params.put("pageSize", pageSize.toString());
        params.put("keyWordsTypeId", keyWordsTypeId);
        params.put("subjectId", subjectId);
        params.put("search", search.toString());
        return HttpUtil.doGet(url, params, null,"utf-8");
    }

    /**
     * 模型信息列表
     *
     * @author lkg
     * @date 2024/4/11
     */
    @GetMapping("/modelList")
    public Result<?> modelList() {
        List<ModelVO> modelVOS = eventService.modelList();
        return Result.OK(modelVOS);
    }

    /**
     * 算法模型信息列表
     *
     * @param subjectId 专题id
     * @param type      类型id
     * @author lkg
     * @date 2024/4/11
     */
    @GetMapping("/algorithmModelList")
    public Object algorithmModelList(@RequestParam String subjectId, @RequestParam Integer type) {
        String url = SERVICE_PROJECT_URL + "event/listNoPage";
        Map<String, String> params = new HashMap<>();
        params.put("subjectId", subjectId);
        params.put("type", type.toString());
        return HttpUtil.doGet(url, params, null,"utf-8");
    }

    /**
     * 模型绑定
     *
     * @author lkg
     * @date 2024/4/11
     */
    @PostMapping("/modelBind")
    public Object modelBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            String url = SERVICE_PROJECT_URL + "event/modelBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 分页列表-新平台管理
     *
     * @param subjectCondition 筛选条件
     * @param pageNo           当前页
     * @param pageSize         返回条数
     * @author lkg
     * @date 2024/4/28
     */
    @GetMapping("/newPlatPageList")
    public Result<?> newPlatPageList(SubjectCondition subjectCondition,
                                     @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                     @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        Page<EventNewPlatVO> pageList = eventService.newPlatPageList(subjectCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }
}
