package com.zzsn.event.llm;


import cn.hutool.http.HttpException;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.dashscope.aigc.generation.Generation;
import com.alibaba.dashscope.aigc.generation.GenerationParam;
import com.alibaba.dashscope.aigc.generation.GenerationResult;
import com.alibaba.dashscope.common.Message;
import com.alibaba.dashscope.common.Role;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * 大模型调用
 *
 * @author lkg
 * @date 2025/7/4
 */
@Service
public class LlmServiceImpl implements LlmService {
    private static final Logger log = LoggerFactory.getLogger(LlmServiceImpl.class);
    @Value("${model.aichain.url:}")
    private String AICHAIN_URL;
    @Value("${model.aichain.authorization:}")
    private String AICHAIN_AUTHOR;
    @Value("${model.aichain.cookie:}")
    private String AICHAIN_COOKIE;

    @Value("${model.qwen.apiKey:}")
    private String QWEN_APIKEY;

    @Value("${model.deepseek.url:}")
    private String DEEP_SEEK_URL;
    @Value("${model.deepseek.api-key:}")
    private String DEEP_SEEK_APIKEY;

    @Value("${model.doubao.url:}")
    private String DOU_BAO_URL;
    @Value("${model.doubao.api-key:}")
    private String DOU_BAO_APIKEY;
    @Value("${model.doubao.default-model:}")
    private String DOU_BAO_MODEL;

    @Override
    public String model(String model, String system, String content) {
        if (StringUtils.isNotEmpty(model)) {
            if (model.startsWith("glm")) {
                return glmModel(model.replace("glm:", ""), system, content);
            } else if (model.startsWith("qwen")) {
                return qwenModel(model.replace("qwen:", ""), system, content);
            } else if (model.startsWith("deepseek")) {
                return deepSeekModel(system, content);
            } else if (model.startsWith("doubao")) {
                return douBaoModel(system, content);
            }
        }
        //默认调用模型
        return glmModel("glm-4-flash", system, content);
    }

    /**
     * glm模型调用
     *
     * @param model   模型
     * @param system  提示词
     * @param content 引用内容
     * @return 大模型响应结果
     */
    public String glmModel(String model, String system, String content) {
        Map<String, Object> params = new HashMap<>();
        params.put("model", model);
        List<Map<String, Object>> messageList = new ArrayList<>();
        Map<String, Object> systemMap = new HashMap<>();
        systemMap.put("role", "system");
        systemMap.put("content", system);
        messageList.add(systemMap);
        Map<String, Object> messagesMap = new HashMap<>();
        messagesMap.put("role", "user");
        messagesMap.put("content", content);
        messageList.add(messagesMap);
        params.put("messages", messageList);
        params.put("history", "[]");
        params.put("top_p", 0.7);
        params.put("temperature", 0.01);
        String responseStr = glmPost(AICHAIN_URL, new JSONObject(params));
        String choices = JSONObject.parseObject(responseStr).getString("choices");
        JSONObject choicesObject = (JSONObject) JSON.parseArray(choices).get(0);
        String messageStr = choicesObject.getString("message");
        return JSONObject.parseObject(messageStr).getString("content");
    }

    /**
     * 千问模型调用
     *
     * @param model   模型
     * @param system  提示词
     * @param content 引用内容
     * @return 大模型响应结果
     */
    public String qwenModel(String model, String system, String content) {
        try {
            Generation gen = new Generation();
            Message systemMsg = Message.builder()
                    .role(Role.SYSTEM.getValue())
                    .content(system)
                    .build();
            Message userMsg = Message.builder()
                    .role(Role.USER.getValue())
                    .content(content)
                    .build();
            GenerationParam param = GenerationParam.builder()
                    // 若没有配置环境变量，请用阿里云百炼API Key将下行替换为：.apiKey("sk-xxx")
                    .apiKey(QWEN_APIKEY)
                    // 模型列表：https://help.aliyun.com/zh/model-studio/getting-started/models
                    .model(model)
                    .messages(Arrays.asList(systemMsg, userMsg))
                    .resultFormat(GenerationParam.ResultFormat.MESSAGE)
                    .build();
            GenerationResult call = gen.call(param);
            return call.getOutput().getChoices().get(0).getMessage().getContent();
        } catch (Exception e) {
            log.error("qwenModel error:{}", e.getMessage(), e);
        }
        return "";
    }

    public String deepSeekModel(String system, String content) {
        try {
            Map<String, Object> params = new HashMap<>();
            params.put("model", "deepseek-chat");
            params.put("stream", false);
            Message systemMsg = Message.builder()
                    .role(Role.SYSTEM.getValue())
                    .content(system)
                    .build();
            Message userMsg = Message.builder()
                    .role(Role.USER.getValue())
                    .content(content)
                    .build();
            List<Message> messages = Arrays.asList(systemMsg, userMsg);
            params.put("messages", messages);
            HttpResponse execute = HttpUtil.createPost(DEEP_SEEK_URL)
                    .header("Content-Type", "application/json")
                    .header("Authorization", "Bearer " + DEEP_SEEK_APIKEY)
                    .body(JSONUtil.toJsonStr(params))
                    .execute();
            return execute.body();
        } catch (HttpException e) {
            log.error("deepseek error:{}", e.getMessage(), e);
        }
        return null;
    }

    public String douBaoModel(String system, String content) {
        try {
            Map<String, Object> params = new HashMap<>();
            params.put("model", DOU_BAO_MODEL);
            Message systemMsg = Message.builder()
                    .role(Role.SYSTEM.getValue())
                    .content(system)
                    .build();
            Message userMsg = Message.builder()
                    .role(Role.USER.getValue())
                    .content(content)
                    .build();
            List<Message> messages = Arrays.asList(systemMsg, userMsg);
            params.put("messages", messages);
            HttpResponse execute = HttpUtil.createPost(DOU_BAO_URL)
                    .header("Content-Type", "application/json")
                    .header("Authorization", "Bearer " + DOU_BAO_APIKEY)
                    .body(JSONUtil.toJsonStr(params))
                    .execute();
            String body = execute.body();
            JSONObject jsonObject = JSONObject.parseObject(body);
            JSONArray choices = jsonObject.getJSONArray("choices");
            JSONObject choice = choices.getJSONObject(0);
            JSONObject message = choice.getJSONObject("message");
            return message.getString("content");
        } catch (HttpException e) {
            log.error("doubao error:{}", e.getMessage(), e);
        }
        return null;
    }

    private String glmPost(String urlstr, JSONObject jsonObject) {
        String responseStr = null;
        try {
            URL url = new URL(urlstr);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod("POST");
            conn.setRequestProperty("Content-Type", "application/json; utf-8");
            conn.setRequestProperty("Accept", "application/json");
            conn.setRequestProperty("Authorization", AICHAIN_AUTHOR);
            conn.setRequestProperty("Cookie", AICHAIN_COOKIE);
            conn.setDoOutput(true);
            String jsonInputString = jsonObject.toJSONString();
            try (OutputStream os = conn.getOutputStream()) {
                byte[] input = jsonInputString.getBytes(StandardCharsets.UTF_8);
                os.write(input, 0, input.length);
            }
            // 检查响应状态
            if (conn.getResponseCode() != HttpURLConnection.HTTP_OK) {
                throw new RuntimeException("Failed : HTTP error code : " + conn.getResponseCode());
            }
            // 读取服务器响应
            BufferedReader br = new BufferedReader(new InputStreamReader(conn.getInputStream()));
            responseStr = br.readLine();
            conn.disconnect();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return responseStr;
    }

}
