package com.zzsn.event.kafka;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.entity.CharacterBasicInfo;
import com.zzsn.event.entity.EventContext;
import com.zzsn.event.entity.EventExpertMap;
import com.zzsn.event.service.CharacterBasicInfoService;
import com.zzsn.event.service.EventContextService;
import com.zzsn.event.service.EventExpertMapService;
import com.zzsn.event.util.CodeGenerateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author lkg
 * 监听消费kafka消息
 * 消费topic中消息(注意：如果监听的topic不存在则会自动创建)
 * @date 2022/7/15
 */
@Slf4j
@Component
public class KafkaConsumer {
    @Autowired
    private EventContextService eventContextService;
    @Autowired
    private CharacterBasicInfoService charBasicInfoService;
    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private EventExpertMapService eventExpertMapService;

    /**
     * 获取-事件脉络-分析结果数据，并入库
     *
     * @param record 接收的kafka数据
     */
    @KafkaListener(topics = {"${kafka.topic.eventContext}"})
    public void eventContext_new(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isEmpty(value)) {
            return;
        }
        List<eventContextKafkaVO> eventContextKafkaList = JSON.parseArray(value, eventContextKafkaVO.class);
        for (eventContextKafkaVO contextKafkaVO : eventContextKafkaList) {
            String uniqueCode = contextKafkaVO.getUniqueCode();
            String eventId = contextKafkaVO.getEventId();
            LambdaQueryWrapper<EventContext> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(EventContext::getEventId, eventId).eq(EventContext::getUniqueCode, uniqueCode);
            EventContext one = eventContextService.getOne(queryWrapper);
            Integer maxSortNumber = eventContextService.getMaxSortNumber(eventId);
            String keyNodeId;
            if (one == null) {
                EventContext keyNode = new EventContext();
                keyNode.setEventId(eventId);
                keyNode.setTitle(contextKafkaVO.getTitle());
                keyNode.setPublishDate(contextKafkaVO.getPublishDate());
                keyNode.setUniqueCode(uniqueCode);
                keyNode.setYnKeyNode(1);
                keyNode.setParentId("0");
                maxSortNumber = maxSortNumber + 1;
                keyNode.setSortNumber(maxSortNumber);
                eventContextService.save(keyNode);
                keyNodeId = keyNode.getId();
            } else {
                keyNodeId = one.getId();
            }
            List<EventContext> saveList = new ArrayList<>();
            List<EventContext> eventContextList = contextKafkaVO.getRelatedArticles();
            if (CollectionUtils.isNotEmpty(eventContextList)) {
                for (int i = 0; i < eventContextList.size(); i++) {
                    EventContext eventContext = eventContextList.get(i);
                    LambdaQueryWrapper<EventContext> query = Wrappers.lambdaQuery();
                    query.eq(EventContext::getEventId, eventId).eq(EventContext::getArticleId, eventContext.getArticleId());
                    int count = eventContextService.count(query);
                    if (count == 0) {
                        eventContext.setEventId(eventId);
                        eventContext.setParentId(keyNodeId);
                        eventContext.setSortNumber(maxSortNumber + i + 1);
                        if (eventContext.getTitle().length() > 500) {
                            eventContext.setTitle(eventContext.getTitle().substring(0, 500));
                        }
                        saveList.add(eventContext);
                    }
                }
                eventContextService.saveBatch(saveList);
            }
        }
        log.info("事件-{}，本次事件脉络更新完成", eventContextKafkaList.get(0).getEventId());
    }

    /**
     * 事件-专家数据
     *
     * @param record 接收的kafka数据
     * @author lkg
     * @date 2025/9/16
     */
    @KafkaListener(topics = {"${kafka.topic.eventExpert}"})
    public void eventExpert(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isEmpty(value)) {
            return;
        }
        JSONObject jsonObject = JSONObject.parseObject(value);
        List<CharacterBasicInfo> eventExpertList = JSON.parseArray(jsonObject.getString("expertList"), CharacterBasicInfo.class);
        if (CollectionUtils.isEmpty(eventExpertList)) {
            return;
        }
        for (CharacterBasicInfo characterBasicInfo : eventExpertList) {
            if (StringUtils.isNotEmpty(characterBasicInfo.getId())) {
                characterBasicInfo = charBasicInfoService.getById(characterBasicInfo.getId());
                characterBasicInfo.setId(null);
                characterBasicInfo.setCreateTime(null);
                characterBasicInfo.setUpdateTime(null);
            } else {
                characterBasicInfo = new CharacterBasicInfo();
                characterBasicInfo.setUid(codeGenerateUtil.geneCharacterUid());
            }
            characterBasicInfo.setCategory("2");
            characterBasicInfo.setTypeId("1966316982441611266");
            LambdaQueryWrapper<CharacterBasicInfo> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(CharacterBasicInfo::getName, characterBasicInfo.getName())
                    .eq(CharacterBasicInfo::getDepartment, characterBasicInfo.getDepartment())
                    .eq(CharacterBasicInfo::getCategory, "2")
                    .eq(CharacterBasicInfo::getTypeId, "1966316982441611266");
            CharacterBasicInfo one = charBasicInfoService.getOne(queryWrapper);
            String expertId;
            if (one == null) {
                charBasicInfoService.save(characterBasicInfo);
                expertId = characterBasicInfo.getId();
            } else {
                expertId = one.getId();
            }
            EventExpertMap eventExpertMap = new EventExpertMap();
            eventExpertMap.setExpertId(expertId);
            eventExpertMap.setEventId(jsonObject.getString("eventId"));
            eventExpertMap.setArticleId(jsonObject.getString("articleId"));
            eventExpertMapService.save(eventExpertMap);
        }
    }
}
