package com.zzsn.event.util;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.alibaba.fastjson2.JSONWriter;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.vo.es.Label;
import com.zzsn.event.vo.es.SpecialInformation;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.core.MainResponse;
import org.elasticsearch.client.indices.CreateIndexRequest;
import org.elasticsearch.client.indices.CreateIndexResponse;
import org.elasticsearch.client.indices.GetIndexRequest;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.query.*;
import org.elasticsearch.index.reindex.BulkByScrollResponse;
import org.elasticsearch.index.reindex.BulkByScrollTask;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.index.reindex.UpdateByQueryRequest;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.script.Script;
import org.elasticsearch.script.ScriptType;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.fetch.subphase.FetchSourceContext;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Es操作相关工具栏
 *
 * @author kongliufeng
 * @create 2020-08-03 16:48
 */
@Slf4j
@Component
public class EsOpUtil<T> {

    @Resource
    RestHighLevelClient client;

    /**
     * 获取节点相关信息
     *
     * @return
     */
    public Map<String, Object> getEsInfo() {
        try {
            Map<String, Object> map = new HashMap<>(100);
            //获取Es相关集群信息
            MainResponse response = client.info(RequestOptions.DEFAULT);
            String clusterName = response.getClusterName();
            String clusterUuid = response.getClusterUuid();
            String nodeName = response.getNodeName();
            MainResponse.Version version = response.getVersion();
            String buildDate = version.getBuildDate();
            String buildFlavor = version.getBuildFlavor();
            String buildHash = version.getBuildHash();
            String buildType = version.getBuildType();
            String luceneVersion = version.getLuceneVersion();
            String minimumIndexCompatibilityVersion = version.getMinimumIndexCompatibilityVersion();
            String minimumWireCompatibilityVersion = version.getMinimumWireCompatibilityVersion();
            String number = version.getNumber();
            map.put("clusterName", clusterName);
            map.put("clusterUuid", clusterUuid);
            map.put("nodeName", nodeName);
            map.put("version", version);
            map.put("buildDate", buildDate);
            map.put("buildFlavor", buildFlavor);
            map.put("buildHash", buildHash);
            map.put("buildType", buildType);
            map.put("luceneVersion", luceneVersion);
            map.put("minimumIndexCompatibilityVersion", minimumIndexCompatibilityVersion);
            map.put("minimumWireCompatibilityVersion", minimumWireCompatibilityVersion);
            map.put("number", number);
            return map;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取低级客户端
     *
     * @return
     */
    public RestClient getLowLevelClient() {
        return client.getLowLevelClient();
    }

    /**
     * 创建索引
     *
     * @param index
     * @return
     */
    public boolean indexCreate(String index) {
        try {
            if (!indexExist(index)) {
                CreateIndexRequest request = new CreateIndexRequest(index);
                CreateIndexResponse createIndexResponse = client.indices().create(request, RequestOptions.DEFAULT);
                log.info(createIndexResponse.isAcknowledged() ? "创建索引[{}]成功" : "创建索引[{}]失败", index);
                return createIndexResponse.isAcknowledged();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * 判断索引是否存在
     *
     * @param indices
     * @return
     */
    @SneakyThrows
    public boolean indexExist(String... indices) {
        GetIndexRequest request = new GetIndexRequest(indices);
        return client.indices().exists(request, RequestOptions.DEFAULT);
    }

    /**
     * 删除索引
     *
     * @param index
     * @return
     */
    @SneakyThrows
    public boolean deleteIndex(String index) {
        DeleteIndexRequest request = new DeleteIndexRequest(index);
        AcknowledgedResponse delete = client.indices().delete(request, RequestOptions.DEFAULT);
        return delete.isAcknowledged();
    }

    /**
     * 判断doc是否存在
     *
     * @param index
     * @param id
     * @return
     */
    @SneakyThrows
    public boolean docExists(String index, String id) {
        GetRequest request = new GetRequest(index, id);
        //禁用提取_source
        request.fetchSourceContext(new FetchSourceContext(false));
        //禁用获取存储的字段
        request.storedFields("_none_");
        boolean exists = client.exists(request, RequestOptions.DEFAULT);
        return exists;
    }


    public Boolean isNotExistUrl(String url, String index) {
        try {
            SearchRequest searchRequest = new SearchRequest(index);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            TermsQueryBuilder sourceAddress = QueryBuilders.termsQuery("sourceAddress", url);
            searchSourceBuilder.query(sourceAddress);
            //不返回文本内容
            searchSourceBuilder.fetchSource(false);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = searchResponse.getHits();
            Long total = hits.getTotalHits().value;
            if (total == 0) {
                return true;
            } else if (total == 1) {
                return false;
            } else if (total > 1) {
                log.warn("url为[{}]在es库中存在[{}]条数据", url, total);
                return false;
            }
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return false;
    }

    /**
     * 批量插入数据
     *
     * @param index 索引库
     * @param list  List<ESBaseData> 批量保存的list,根据实际需求实体集成ESBaseData
     */
    public void docSaveBulk(String index, List<T> list) {
        BulkRequest request = new BulkRequest();
        request.timeout(TimeValue.timeValueMinutes(10));
        for (int i = 0; i < list.size(); i++) {
            T b = list.get(i);
            String id;
            try {
                Field field = b.getClass().getDeclaredField("id");
                field.setAccessible(true);
                id = (String) field.get(b);
            } catch (NoSuchFieldException e) {
                log.info("实体没有id字段");
                continue;
            } catch (IllegalAccessException e) {
                log.info("无权限访问id字段");
                continue;
            }
            request.add(new IndexRequest(index).id(id).source(
                    JSON.toJSONString(list.get(i)), XContentType.JSON
            ));
        }
        try {
            BulkResponse bulk = client.bulk(request, RequestOptions.DEFAULT);
            BulkItemResponse[] bulkItemResponses = bulk.getItems();
            for (BulkItemResponse response : bulkItemResponses) {
                if (response.isFailed()) {
                    log.info("批量保存[{}]过程中,id为[{}]的保存失败,失败原因[{}]", response.getIndex(), response.getId(), response.getFailureMessage());
                } else {
                    log.info("批量保存[{}]过程中,id为[{}]的保存成功,状态[{}],version[{}]", response.getIndex(), response.getId(), response.status(), response.getVersion());
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
            log.warn("批量[{}]保存失败", index);
        }
    }

    /**
     * 批量插入数据(异步)
     *
     * @param index 索引库
     * @param list  List<ESBaseData> 批量保存的list,根据实际需求实体集成ESBaseData
     */
    public void docSavaBulkAsync(String index, List<T> list) {
        BulkRequest request = new BulkRequest();
        request.timeout(TimeValue.timeValueMinutes(10));
        for (int i = 0; i < list.size(); i++) {
            T b = list.get(i);

            String id = null;
            try {
                Field field = b.getClass().getDeclaredField("id");
                field.setAccessible(true);
                id = (String) field.get(b);
            } catch (NoSuchFieldException e) {
                log.info("实体没有id字段");
                continue;
            } catch (IllegalAccessException e) {
                log.info("无权限访问id字段");
                continue;
            }
            request.add(new IndexRequest(index).id(id).source(
                    JSON.toJSONString(list.get(i)), XContentType.JSON
            ));
        }
        client.bulkAsync(request, RequestOptions.DEFAULT, new ActionListener<BulkResponse>() {
            @Override
            public void onResponse(BulkResponse bulkItemResponses) {
                BulkItemResponse[] bulkItems = bulkItemResponses.getItems();
                int length = bulkItems.length;
                for (int i = 0; i < length; ++i) {
                    BulkItemResponse response = bulkItems[i];
                    if (response.isFailed()) {//查看所有请求失败结果
                        log.info("批量保存[{}]过程中,id为[{}]的保存失败,失败原因[{}]", response.getIndex(), response.getId(), response.getFailureMessage());
                    } else {//请求成功的
                        log.info("批量保存[{}]过程中,id为[{}]的保存成功,状态[{}],version[{}]", response.getIndex(), response.getId(), response.status(), response.getVersion());
                    }
                }
            }

            @Override
            public void onFailure(Exception e) {
                log.warn("批量[{}]保存失败,失败原因[{}]", index, e.getMessage());
            }
        });
    }

    /**
     * 插入数据
     *
     * @param index
     * @param id
     * @param object
     * @return
     */
    public String docSavaByEntity(String index, String id, Object object) {
        return docSaveByJson(index, id, JSON.toJSONString(object, JSONWriter.Feature.WriteMapNullValue));
    }


    public void docSavaByEntityAsync(String index, String id, Object object) {
        docSaveByJsonAsync(index, id, JSON.toJSONString(object, JSONWriter.Feature.WriteMapNullValue));
    }

    /**
     * 保存json
     *
     * @param index
     * @param id
     * @param jsonStr
     * @return
     */
    public String docSaveByJson(String index, String id, String jsonStr) {
        try {
            IndexRequest request = new IndexRequest(index)
                    .id(id)
                    .source(jsonStr, XContentType.JSON);
            request.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
            IndexResponse indexResponse = client.index(request, RequestOptions.DEFAULT);
            return indexResponse.getId();
        } catch (IOException e) {
            log.warn("同步保存doc失败, _index=[{}], _id=[{}]", index, id);
        }
        return index;
    }

    /**
     * 异步创建doc
     *
     * @param index
     * @param id
     * @param jsonStr
     * @return
     */
    public void docSaveByJsonAsync(String index, String id, String jsonStr) {
        IndexRequest request = new IndexRequest(index);
        request.id(id);
        request.source(jsonStr, XContentType.JSON);
        client.indexAsync(request, RequestOptions.DEFAULT, new ActionListener<IndexResponse>() {
            @Override
            public void onResponse(IndexResponse indexResponse) {
                log.info("异步保存doc, _index=[{}], _id=[{}]成功, _version=[{}], _result=[{}]", index, indexResponse.getId(), indexResponse.getVersion(), indexResponse.getResult());
            }

            @Override
            public void onFailure(Exception e) {
                e.printStackTrace();
                log.warn("异步保存失败，尝试同步方式保存doc, ex=[{}]", e.getMessage());
                try {
                    IndexResponse response = client.index(request, RequestOptions.DEFAULT);
                    DocWriteResponse.Result result = response.getResult();
                    if (!(result == DocWriteResponse.Result.UPDATED || result == DocWriteResponse.Result.CREATED)) {
                        log.warn("同步保存doc失败,_index=[{}], _id=[{}], _body=[{}]", index, id, jsonStr);
                    }
                } catch (IOException io) {
                    io.printStackTrace();
                }
            }
        });
    }

    public String docSaveByMap(String index, String id, Map<String, Object> map) {
        try {
            IndexRequest request = new IndexRequest(index).id(id)
                    .source(map);
            IndexResponse indexResponse = client.index(request, RequestOptions.DEFAULT);
            return indexResponse.getId();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return index;
    }

    /**
     * 插入数据
     *
     * @param index
     * @param id
     * @param object
     * @return
     */
    public String docSaveByJsonObject(String index, String id, JSONObject object) {
        IndexRequest request = new IndexRequest(index);
        request.id(id);
        try {
            request.source(JSON.toJSONString(object), XContentType.JSON);
            IndexResponse indexResponse = client.index(request, RequestOptions.DEFAULT);
            return indexResponse.getId();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 根据id删除doc
     *
     * @param index
     * @param id
     * @return
     */
    public Boolean docDeleteById(String index, String id) {
        try {
            DeleteRequest deleteRequest = new DeleteRequest(index, id);
            DeleteResponse delete = client.delete(deleteRequest, RequestOptions.DEFAULT);
            if (delete.status() == RestStatus.OK) {
                log.info("DELETE /{}/_doc/{}/\r\n", index, id);
                return true;
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return false;
    }

    public void docDeleteByIdAsync(String index, String id) {
        DeleteRequest request = new DeleteRequest(index, id);
        try {
            client.deleteAsync(request, RequestOptions.DEFAULT, new ActionListener<DeleteResponse>() {
                @Override
                public void onResponse(DeleteResponse deleteResponse) {
                    log.info("删除doc成功, _index=[{}], _id=[{}]", index, deleteResponse.getId());
                }

                @Override
                public void onFailure(Exception e) {
                    e.printStackTrace();
                    log.warn("删除doc失败, _index=[{}], _id=[{}]", index, id);
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 根据条件删除
     *
     * @param index
     * @param query
     * @return
     */
    public Long docDeleteByQuery(final String index, QueryBuilder query) {
        try {
            DeleteByQueryRequest request = new DeleteByQueryRequest(index);
            request.setQuery(query).setRefresh(true);
            BulkByScrollResponse bulkByScrollResponse = client.deleteByQuery(request, RequestOptions.DEFAULT);
            return bulkByScrollResponse.getDeleted();
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    public boolean docUpdateById(String index, String id, String jsonStr) {
        UpdateRequest request = new UpdateRequest(index, id);
        //刷新策略，默认
        request.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        request.setRefreshPolicy("true");
        request.doc(jsonStr, XContentType.JSON);
        try {
            UpdateResponse response = client.update(request, RequestOptions.DEFAULT);
            return response.status() == RestStatus.OK;
        } catch (IOException e) {
            e.printStackTrace();
            log.warn("更新doc失败, _index=[{}], _id=[{}],_jsonStr=[{}]", index, id, jsonStr);
        }
        return false;
    }


    public void docUpdateBulk(String index, List<SpecialInformation> dataList) {
        BulkRequest bulkRequest = new BulkRequest();
        for (SpecialInformation information : dataList) {
            UpdateRequest request = new UpdateRequest(index, information.getId());
            request.doc(JSON.toJSONString(information), XContentType.JSON);
            bulkRequest.add(request);
        }
        bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        // 执行更新请求
        client.bulkAsync(bulkRequest, RequestOptions.DEFAULT, new ActionListener<BulkResponse>() {
            @Override
            public void onResponse(BulkResponse bulkItemResponses) {
                BulkItemResponse[] bulkItems = bulkItemResponses.getItems();
                for (BulkItemResponse response : bulkItems) {
                    if (response.isFailed()) {//查看所有请求失败结果
                        log.info("批量更新[{}]过程中,id为[{}]的更新失败,失败原因[{}]", response.getIndex(), response.getId(), response.getFailureMessage());
                    } else {//请求成功的
                        log.info("批量更新[{}]过程中,id为[{}]的更新成功,状态[{}],version[{}]", response.getIndex(), response.getId(), response.status(), response.getVersion());
                    }
                }
            }

            @Override
            public void onFailure(Exception e) {
                log.warn("批量[{}]更新失败,失败原因[{}]", index, e.getMessage());
            }
        });
    }

    /**
     * 批量删除
     *
     * @param removeList 删除参数
     * @author lkg
     * @date 2024/12/19
     */
    public void docRemoveBulk(List<Map<String, Object>> removeList) {
        BulkRequest bulkRequest = new BulkRequest();
        for (Map<String, Object> m : removeList) {
            String index = m.get("index").toString();
            String id = m.get("id").toString();
            DeleteRequest deleteRequest = new DeleteRequest(index, id);
            bulkRequest.add(deleteRequest);
        }
        bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        try {
            client.bulk(bulkRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            log.error("数据删除失败!", e);
        }
    }

    /**
     * 批量删除
     *
     * @param removeMap 删除参数
     * @author lkg
     * @date 2024/12/19
     */
    public void docRemoveBulk(Map<String, List<SpecialInformation>> removeMap) {
        BulkRequest bulkRequest = new BulkRequest();
        for (Map.Entry<String, List<SpecialInformation>> entry : removeMap.entrySet()) {
            String index = entry.getKey();
            for (SpecialInformation information : entry.getValue()) {
                DeleteRequest deleteRequest = new DeleteRequest(index, information.getId());
                bulkRequest.add(deleteRequest);
            }
        }
        bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        try {
            client.bulk(bulkRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            log.error("数据删除失败!", e);
        }
    }


    /**
     * @Description 判断该专题下的内容是否重复导入
     * @author kongliufeng
     * @创建时间 2020/9/11 18:05
     * @Version 1.0
     */
    public Boolean isExistSubjectAndArticle(String index, String id, String subjectId) {
        try {
            SearchRequest searchRequest = new SearchRequest(index);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
            MatchQueryBuilder matchQueryBuilder = QueryBuilders.matchQuery("subjectId", subjectId);
            boolQueryBuilder.must(matchQueryBuilder);
            MatchQueryBuilder matchQueryBuilder1 = QueryBuilders.matchQuery("id", id);
            boolQueryBuilder.must(matchQueryBuilder1);
            searchSourceBuilder.query(boolQueryBuilder);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = searchResponse.getHits();
            long total = hits.getTotalHits().value;
            if (total > 0) {
                log.info("isExistSubjectAndArticle[index:{}][id:{}][subject:{}]重复,库中已存在", index, id, subjectId);
                return true;
            }
        } catch (IOException e) {
            e.printStackTrace();
            log.info("isExistSubjectAndArticle[index:{}][id:{}][subject:{}]发生异常", index, id, subjectId);
            return false;
        }
        return false;
    }

    public boolean existBySourceAddress(String index, String sourceAddress) {
        SearchRequest searchRequest = new SearchRequest(index);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("sourceAddress.keyword", sourceAddress));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        long count = 0;
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            count = searchResponse.getHits().getTotalHits().value;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return count == 0;
    }

    public boolean existByTitle(String index, String title) {
        SearchRequest searchRequest = new SearchRequest(index);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("title.keyword", title));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        long count = 0;
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            count = searchResponse.getHits().getTotalHits().value;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return count == 0;
    }


    /**
     * 批量更新操作，根据指定的查询条件和字段信息，更新符合条件的文档的指定字段值。
     * 方法名：updataBatchByQuery，表示批量更新文档的方法。
     * 参数：index，指定要更新的索引名称。
     * 参数：boolQuery，BoolQueryBuilder对象，表示更新文档的查询条件。
     * 参数：modifyColum，表示要更新的字段名。
     * 参数：modifyColumValue，表示要更新的字段值。
     * 方法抛出了IOException和InterruptedException异常。
     * 创建了一个UpdateByQueryRequest对象，传入了要更新的索引名称。
     * 设置查询条件，使用setQuery方法，传入BoolQueryBuilder对象。
     * 设置更新脚本，使用setScript方法，传入Script对象。脚本使用painless语言，通过ctx._source.字段名 = 字段值的方式来更新文档的指定字段值。
     * 调用client.updateByQuery方法，传入UpdateByQueryRequest对象和默认的RequestOptions，执行批量更新操作，并返回BulkByScrollResponse对象。
     */
    public void updateBatchByQuery(String index, BoolQueryBuilder boolQuery, String modifyColum, String modifyColumValue) throws IOException, InterruptedException {
        UpdateByQueryRequest request = new UpdateByQueryRequest(index);
        request.setQuery(boolQuery);
        request.setScript(new Script(ScriptType.INLINE, "painless", "ctx._source." + modifyColum + " = '" + modifyColumValue + "'", Collections.emptyMap()));
        BulkByScrollResponse response = client.updateByQuery(request, RequestOptions.DEFAULT);
        long updated = response.getUpdated();
        log.info("更新条数{}", updated);
    }

    /**
     * 批量更新(参数deleteParams 中的key要和es中的字段保持一致)
     * <p>
     * 批量更新操作，根据指定的查询条件和多个字段的映射关系，更新符合条件的文档的多个字段的值。
     * 方法名：updataMoreColumBatchByQuery，表示批量更新文档的多个字段的方法。
     * 参数：index，指定要更新的索引名称。
     * 参数：boolQuery，BoolQueryBuilder对象，表示更新文档的查询条件。
     * 参数：modifyColum_value，表示要更新的多个字段名和对应的字段值的映射关系。
     * 方法抛出了IOException和InterruptedException异常。
     * 创建了一个UpdateByQueryRequest对象，传入了要更新的索引名称。
     * 设置查询条件，使用setQuery方法，传入BoolQueryBuilder对象。
     * 调用getIdOrCode方法，传入多个字段名和字段值的映射关系，获取更新脚本。
     * 设置更新脚本，使用setScript方法，传入Script对象。脚本使用painless语言，通过ctx._source.字段名 = 字段值的方式来更新文档的多个字段的值。
     * 调用client.updateByQuery方法，传入UpdateByQueryRequest对象和默认的RequestOptions，执行批量更新操作，并返回BulkByScrollResponse对象。
     * Collections.emptyMap()是setScript脚本的参数，此方法没有设置其他参数，使用一个空的map
     *
     * @param index        索引
     * @param boolQuery    es查询条件
     * @param modifyParams 更新参数
     * @author lkg
     * @date 2025/4/22
     */
    public void batchUpdateScript(String[]  index, BoolQueryBuilder boolQuery, Map<String, Object> modifyParams) {
        UpdateByQueryRequest request = new UpdateByQueryRequest(index);
        request.setQuery(boolQuery);
        StringBuffer script = new StringBuffer();
        modifyParams.forEach((colum, value) -> {
            if (value instanceof String) {
                script.append("ctx._source.").append(colum).append(" = '").append(value).append("';");
            } else {
                script.append("ctx._source.").append(colum).append(" = ").append(value).append(";");
            }
        });
        request.setScript(new Script(ScriptType.INLINE, "painless", script.toString(), Collections.emptyMap()));
        //可跳过版本冲突的文档
        request.setConflicts("proceed");
        try {
            BulkByScrollResponse response = client.updateByQuery(request, RequestOptions.DEFAULT);
            long updated = response.getUpdated();
            log.info("更新条数{}", updated);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 批量替换字符串(更新的一种)
     *
     * @param index       索引
     * @param boolQuery   es查询条件
     * @param modifyField 编辑的字段集合
     * @param oldValue    旧字符串
     * @param newValue    新字符串
     * @author lkg
     * @date 2025/4/22
     */
    public void batchReplaceScript(String[]  index, BoolQueryBuilder boolQuery, List<String> modifyField, String oldValue, @NotNull String newValue) {
        UpdateByQueryRequest request = new UpdateByQueryRequest(index);
        request.setQuery(boolQuery);
        String script = modifyField.stream()
                .map(field -> String.format(
                        "if(ctx._source.%s != null) { ctx._source.%s = ctx._source.%s.replace('%s', '%s'); }",
                        field, field, field, oldValue, newValue))
                .collect(Collectors.joining(""));
        request.setScript(new Script(ScriptType.INLINE, "painless", script, Collections.emptyMap()));
        //可跳过版本冲突的文档
        request.setConflicts("proceed");
        try {
            BulkByScrollResponse response = client.updateByQuery(request, RequestOptions.DEFAULT);
            long updated = response.getUpdated();
            log.info("更新条数{}", updated);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }


    /**
     * 批量删除nest字段中的数据
     * <p>
     * 参数deleteParams 中的key要和es中的字段保持一致
     *
     * @param index        索引
     * @param boolQuery    es查询条件
     * @param deleteParams 删除条件
     * @param nestField    nest字段
     * @author lkg
     * @date 2025/4/22
     */
    public void batchNestedDeleteScript(String[] index, BoolQueryBuilder boolQuery, Map<String, Object> deleteParams, String nestField) {
        UpdateByQueryRequest request = new UpdateByQueryRequest(index);
        request.setQuery(boolQuery);
        //Painless 脚本：从 nested 数组中删除匹配项
        String scriptSource = "ctx._source." + nestField + ".removeIf(item -> ";
        StringBuilder paramString = new StringBuilder();
        deleteParams.forEach((colum, value) -> {
            if (value instanceof String) {
                paramString.append(" && ").append("item.").append(colum).append(" == params.").append(colum);
            } else {
                paramString.append(" && ").append("item.").append(colum).append(" == params.").append(colum);
            }
        });
        String substring = paramString.substring(4);
        scriptSource = scriptSource + substring + ")";
        Script script = new Script(ScriptType.INLINE, "painless", scriptSource, deleteParams);
        request.setScript(script);
        //可跳过版本冲突的文档
        request.setConflicts("proceed");
        try {
            BulkByScrollResponse response = client.updateByQuery(request, RequestOptions.DEFAULT);
            long deleted = response.getUpdated();
            log.info("更新条数{}", deleted);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 批量新增nest字段中的数据
     * <p>
     * 参数addParams 中的key要和es中的字段保持一致
     *
     * @param index       索引
     * @param boolQuery   es查询条件
     * @param addParams   添加的数据
     * @param nestField   nest字段
     * @param uniqueField 唯一字段
     * @author lkg
     * @date 2025/4/22
     */
    public void batchNestedAddScript(String[]  index, BoolQueryBuilder boolQuery, Map<String, Object> addParams, String nestField, String uniqueField) {
        UpdateByQueryRequest request = new UpdateByQueryRequest(index);
        request.setQuery(boolQuery);
        //Painless 脚本：从 nested 数组中删除匹配项
        String scriptString = "if(ctx._source.NEST_FIELD==null){List ls=new ArrayList();ls.add(params.NEST_FIELD);ctx._source.NEST_FIELD=ls;}else{List olds=new ArrayList();ctx._source.NEST_FIELD.forEach(item->{olds.add(item.UNIQUE_FIELD);});if(!olds.contains(params.NEST_FIELD.UNIQUE_FIELD)){ctx._source.NEST_FIELD.add(params.NEST_FIELD);}}";
        String scriptSource = scriptString.replace("NEST_FIELD", nestField).replace("UNIQUE_FIELD", uniqueField);
        Script script = new Script(ScriptType.INLINE, "painless", scriptSource, addParams);
        request.setScript(script);
        //可跳过版本冲突的文档
        request.setConflicts("proceed");
        try {
            BulkByScrollResponse response = client.updateByQuery(request, RequestOptions.DEFAULT);
            long deleted = response.getUpdated();
            log.info("更新条数{}", deleted);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }


    /**
     * 这段代码是一个用于获取Elasticsearch索引中某个字段的最大值的方法。
     * getColumMax方法：
     * 参数：index表示索引名称，boolQuery表示查询条件，colum表示要获取最大值的字段名。
     * 创建一个搜索请求SearchRequest，并将index作为参数传入。
     * 创建一个搜索源构建器SearchSourceBuilder，并设置其大小为0（即只返回聚合结果，不返回具体文档）。
     * 如果boolQuery不为空，则将其设置为搜索源构建器的查询条件。
     * 创建一个TermsAggregationBuilder聚合器，用于按照字段colum进行聚合。
     * 设置聚合器的大小为1，表示只返回一个聚合结果。
     * 设置聚合器的排序方式为按照聚合桶的键值降序排序。
     * 将聚合器添加到搜索源构建器中。
     * 将搜索源构建器设置为搜索请求的源。
     * 使用client.search方法执行搜索请求，返回一个SearchResponse对象。
     * 从SearchResponse中获取聚合结果Aggregations。
     * 根据聚合结果中的聚合器名称"groupByColum"获取到对应的Terms聚合器。
     * 从Terms聚合器中获取聚合桶codeBuckets，即按照字段colum聚合后的结果。
     * 如果聚合桶不为空，则返回第一个聚合桶的键值转换为字符串；否则返回null。
     * 如果执行搜索请求过程中发生异常，则打印异常堆栈并返回null。
     * 该方法的作用是执行一个查询请求，按照指定字段进行聚合，并获取聚合结果中的最大值。返回的最大值是一个字符串类型。
     */

    public String getColumMax(String index, BoolQueryBuilder boolQuery, String colum) {
        SearchRequest searchRequest = new SearchRequest(index);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.size(0);
        if (ObjectUtil.isNotEmpty(boolQuery)) {
            searchSourceBuilder.query(boolQuery);
        }
        TermsAggregationBuilder aggregation = AggregationBuilders.terms("groupByColum")
                .field(colum)
                .size(1)
                .order(BucketOrder.key(false));
        searchSourceBuilder.aggregation(aggregation);
        searchRequest.source(searchSourceBuilder);

        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = response.getAggregations();
            Terms groupCode = aggregations.get("groupByColum");
            List<? extends Terms.Bucket> codeBuckets = groupCode.getBuckets();
            return CollectionUtil.isNotEmpty(codeBuckets) ? codeBuckets.get(0).getKeyAsString() : null;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }


    /**
     * 根据文档id更新
     * 参数：index表示索引名称，id表示文档的id，args表示要更新的字段和对应的值。
     * 首先判断args是否为空，如果为空则直接返回。
     * 创建一个更新请求UpdateRequest，并将index和id作为参数传入。
     * 使用XContentBuilder构建要更新的内容，将args中的字段和值添加到contentBuilder中。
     * 将contentBuilder设置为更新请求的内容。
     * 使用client.update方法执行更新请求，返回一个UpdateResponse对象。
     * 根据UpdateResponse的结果进行处理，如果更新成功，则打印日志；如果没有进行任何更改，则打印日志；如果更新失败，则打印日志。
     */
    public void updateById(String index, String id, Map<String, Object> args) {
        if (CollUtil.isEmpty(args)) {
            return;
        }
        // 执行更新请求
        try {
            UpdateResponse response = client.update(createUpdateRequest(index, id, args, true), RequestOptions.DEFAULT);
            RestStatus status = response.status();
            if (status.getStatus() != 200) {
                log.info("{},更新失败", id);
            }
        } catch (IOException e) {
            log.info("{},更新异常", id);
        }
    }

    /**
     * 根据id更新，批量更新
     * 参数：index表示索引名称，batch表示要批量更新的文档，其中batch是一个Map，key为文档的id，value为要更新的字段和对应的值。
     * 首先判断batch是否为空，如果为空则直接返回。
     * 创建一个批量请求BulkRequest。
     * 遍历batch中的每个文档，将每个文档的id和要更新的字段和值创建一个更新请求UpdateRequest，并将其添加到bulkRequest中。
     * 使用client.bulk方法执行批量更新请求，返回一个BulkResponse对象。
     * 根据BulkResponse的结果进行处理，如果有更新失败的情况，则打印日志；如果全部更新成功，则打印日志。
     */
    public void bulkUpdateFields(String index, Map<String, Map<String, Object>> args) {
        if (CollUtil.isEmpty(args)) {
            return;
        }
        BulkRequest bulkRequest = new BulkRequest();
        args.forEach((id, args1) -> {
            try {
                bulkRequest.add(createUpdateRequest(index, id, args1, false));
            } catch (IOException e) {
                e.printStackTrace();
            }
        });
        // 执行更新请求
        try {
            BulkResponse bulk = client.bulk(bulkRequest, RequestOptions.DEFAULT);
            BulkItemResponse[] bulkItemResponses = bulk.getItems();
            for (BulkItemResponse response : bulkItemResponses) {
                if (response.isFailed()) {
                    log.info("批量更新字段[{}]过程中,id为[{}]的更新失败,失败原因[{}]", response.getIndex(), response.getId(), response.getFailureMessage());
                } else {
                    log.info("批量保存[{}]过程中,id为[{}]的保存成功,状态[{}],version[{}]", response.getIndex(), response.getId(), response.status(), response.getVersion());
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 创建更新请求对象
     * 参数：index表示索引名称，documentId表示文档的id，args表示要更新的字段和对应的值。
     * 创建一个更新请求UpdateRequest，并将index和documentId作为参数传入。
     * 使用XContentBuilder构建要更新的内容，将args中的字段和值添加到contentBuilder中。
     * 将contentBuilder设置为更新请求的内容。
     * 返回更新请求UpdateRequest对象。
     */
    private UpdateRequest createUpdateRequest(String index, String documentId, Map<String, Object> args, Boolean refreshPolicy) throws IOException {
        UpdateRequest request = new UpdateRequest(index, documentId);
        if (Boolean.TRUE.equals(refreshPolicy)) {
            //刷新策略，立即刷新
            request.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        }
        // 创建要更新的内容
        XContentBuilder contentBuilder = XContentFactory.jsonBuilder();
        contentBuilder.startObject();
        args.forEach((fieldName, fieldValue) ->
                {
                    try {
                        contentBuilder.field(fieldName, fieldValue);
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
        );
        contentBuilder.endObject();
        // 设置更新请求的内容
        request.doc(contentBuilder);
        return request;
    }


    /**
     * 根据id查询各类资讯详情
     */
    public T getInfoById(String index, String id, Class<T> entry) {
        try {
            SearchRequest searchRequest = new SearchRequest(index);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            TermQueryBuilder termQueryBuilder = QueryBuilders.termQuery("id", id);
            searchSourceBuilder.query(termQueryBuilder);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = searchResponse.getHits().getHits();
            T res = null;
            if (hits.length > 0) {
                res = JSON.parseObject(hits[0].getSourceAsString(), entry);
            }
            return res;
        } catch (IOException e) {
            log.info("查询异常{}", e.getMessage(), e);
            return null;
        }
    }

    /**
     * 通用查询
     */
    public List<T> queryList(String index, QueryBuilder queryBuilder, Class<T> entry) {
        List<T> list = new ArrayList<>();

        try {
            SearchRequest searchRequest = new SearchRequest(index);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            searchSourceBuilder.query(queryBuilder);
            searchRequest.source(searchSourceBuilder);

            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            // 处理搜索结果
            SearchHit[] hits = searchResponse.getHits().getHits();
            if (hits.length > 0) {
                Arrays.stream(hits).forEach(e -> {
                    T t = JSON.parseObject(e.getSourceAsString(), entry);
                    list.add(t);
                });
            }
            return list;
        } catch (IOException e) {
            log.info("查询异常{}", e.getMessage(), e);
            return list;
        }
    }

    /**
     * 通用分页查询
     */
    public Page<T> queryPage(String index, QueryBuilder queryBuilder, Class<T> entry, Integer pageNo, Integer pageSize) {
        List<T> list = new ArrayList<>();
        Page<T> pageData = new Page<>(pageNo, pageSize);
        try {
            SearchRequest searchRequest = new SearchRequest(index);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            //设置分页参数
            searchSourceBuilder.size(pageSize);
            searchSourceBuilder.from((pageNo - 1) * pageSize);
            //默认最大数量是10000，设置为true后，显示准确数量
            searchSourceBuilder.trackTotalHits(true);
            searchSourceBuilder.sort(SortBuilders.fieldSort("publishDate").order(SortOrder.DESC));
            searchSourceBuilder.query(queryBuilder);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            // 处理搜索结果
            SearchHit[] hits = searchResponse.getHits().getHits();
            if (hits.length > 0) {
                Arrays.stream(hits).forEach(e -> {
                    T t = JSON.parseObject(e.getSourceAsString(), entry);
                    list.add(t);
                });
            }
            pageData.setTotal(searchResponse.getHits().getTotalHits().value);
            pageData.setRecords(list);
            return pageData;
        } catch (IOException e) {
            log.info("查询异常{}", e.getMessage(), e);
            return pageData;
        }
    }
}
