package com.zzsn.event.util;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 本地缓存，带时效性
 *
 * @author lkg
 * @date 2025/6/4
 */
public class LocalCache<K, V> {

    private final ConcurrentHashMap<K, CacheValue<V>> cache;
    private final long defaultExpiryTime; // 默认过期时间(毫秒)

    public LocalCache(long defaultExpiryTime) {
        this.cache = new ConcurrentHashMap<>();
        this.defaultExpiryTime = defaultExpiryTime;
    }

    public void put(K key, V value) {
        put(key, value, defaultExpiryTime);
    }

    public void put(K key, V value, long expiryTime) {
        cache.put(key, new CacheValue<>(value, System.currentTimeMillis() + expiryTime));
    }

    public V get(K key) {
        CacheValue<V> cacheValue = cache.get(key);
        if (cacheValue == null || cacheValue.isExpired()) {
            cache.remove(key);
            return null;
        }
        return cacheValue.getValue();
    }

    public void remove(K key) {
        cache.remove(key);
    }

    public void clear() {
        cache.clear();
    }

    public int size() {
        return cache.size();
    }


    static class CacheValue<V> {
        private final V value;
        private final long expiryTime;

        CacheValue(V value, long expiryTime) {
            this.value = value;
            this.expiryTime = expiryTime;
        }

        V getValue() {
            return value;
        }

        boolean isExpired() {
            return System.currentTimeMillis() > expiryTime;
        }
    }

    public static void main(String[] args) throws InterruptedException {
        LocalCache<String, String> cache = new LocalCache<>(2 * 1000);

        cache.put("key1", "value1");
        System.out.println("Get key1 immediately: " + cache.get("key1"));

        TimeUnit.SECONDS.sleep(1); // 等待3秒
        System.out.println("Get key1 after 3 seconds: " + cache.get("key1"));
    }
}
