package com.zzsn.event.task;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.Event;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.AnalysisService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.vo.CoOccurrenceVO;
import com.zzsn.event.vo.PropagationPathVo;
import com.zzsn.event.vo.SubjectKafkaVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * @author lkg
 * @description: 专题分析定时任务
 * @date 2022/7/8 11:58
 */
@Slf4j
@Component
public class AnalysisTask {

    @Autowired
    private IEventService eventService;
    @Autowired
    private EsService esService;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private AnalysisService analysisService;


    /**
     * 重新生成事件分析(版本)
     * 每天凌晨00:30执行一次
     *
     * @author lkg
     * @date 2025/7/9
     */
    //@Scheduled(cron = "0 30 0 * * ?")
    public void analysis() {
        Date now = new Date();
        Date deadlineDate = DateUtil.addDate(now, -1);
        LambdaQueryWrapper<Event> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(Event::getStatus, 1).gt(Event::getCreateTime, "2025-07-27");
        List<Event> eventList = eventService.list(queryWrapper);
        for (Event event : eventList) {
            CompletableFuture.runAsync(() -> {
                String eventId = event.getId();
                String estimateStatus = event.getEstimateStatus();
                if (StringUtils.isNotBlank(estimateStatus)
                        && !estimateStatus.equals("已完成") && !estimateStatus.equals("未启用")) {
                    if (DateUtil.stringToDate(estimateStatus, "yyyy-MM-dd HH:mm:ss").compareTo(deadlineDate) > 0) {
                        analysisService.regenerate(eventId);
                    }
                }
            });
        }
    }

    /**
     * 定时生成传播路径
     * 每天凌晨0点执行一次
     */
    @Scheduled(cron = "0 0 0 * * ?")
    public void propagationPath() {
        Date today = new Date();
        Date deadlineDate = DateUtil.addDate(today, -1);
        LambdaQueryWrapper<Event> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(Event::getStatus, 1);
        List<Event> eventList = eventService.list(queryWrapper);
        for (Event event : eventList) {
            CompletableFuture.runAsync(() -> {
                String eventId = event.getId();
                int count = esService.count(eventId, null, null);
                if (count > 0) {
                    String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.PROPAGATION_KEY + eventId;
                    Date timeDisable = event.getEndTime();
                    //已经结束的事件专题，永久缓存
                    if (timeDisable != null && deadlineDate.compareTo(timeDisable) > 0) {
                        Object cacheObject = redisUtil.get(key);
                        if (cacheObject == null) {
                            PropagationPathVo pathVo = analysisService.propagationPath(eventId);
                            if (ObjectUtils.isNotEmpty(pathVo)) {
                                redisUtil.set(key, pathVo);
                                log.info("事件-{},传播路径数据【永久】缓存成功!", event.getEventName());
                            }
                        }
                    } else {//已经结束的事件专题，缓存有效期一天
                        PropagationPathVo pathVo = analysisService.propagationPath(eventId);
                        if (ObjectUtils.isNotEmpty(pathVo)) {
                            redisUtil.set(key, pathVo, 3600 * 24);
                            log.info("事件-{},传播路径数据缓存成功!", event.getEventName());
                        }
                    }
                }
            });
        }
    }

    /**
     * 定时缓存热词共现关系数据
     * 每天凌晨0点10分执行一次
     */
    @Scheduled(cron = "0 10 0 * * ?")
    public void co_occurrence() {
        Date today = new Date();
        Date deadlineDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> events = eventService.eventSubjectList();
        for (SubjectKafkaVo event : events) {
            CompletableFuture.runAsync(() -> {
                String eventId = event.getId();
                String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.CO_OCCURRENCE + eventId;
                Date timeDisable = event.getTimeDisable();
                if (timeDisable != null && deadlineDate.compareTo(timeDisable) > 0) {
                    Object cacheObject = redisUtil.get(key);
                    if (cacheObject == null) {
                        List<CoOccurrenceVO> occurrenceList = analysisService.coOccurrence(eventId, null, null);
                        if (CollectionUtils.isNotEmpty(occurrenceList)) {
                            redisUtil.set(key, occurrenceList);
                            log.info("专题-{},热词共现关系数据【永久】缓存成功!", event.getSubjectName());
                        }
                    }
                } else {
                    List<CoOccurrenceVO> occurrenceList = analysisService.coOccurrence(eventId, null, null);
                    if (CollectionUtils.isNotEmpty(occurrenceList)) {
                        redisUtil.set(key, occurrenceList, 3600 * 24);
                        log.info("专题-{},热词共现关系数据缓存成功!", event.getSubjectName());
                    }
                }
            });
        }
    }
}
