package com.zzsn.event.llm;


import cn.hutool.http.HttpException;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.dashscope.aigc.generation.Generation;
import com.alibaba.dashscope.aigc.generation.GenerationParam;
import com.alibaba.dashscope.aigc.generation.GenerationResult;
import com.alibaba.dashscope.common.Message;
import com.alibaba.dashscope.common.Role;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.zzsn.event.config.properties.LlmProperties;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * 大模型调用
 *
 * @author lkg
 * @date 2025/7/4
 */
@Slf4j
@Service
public class LlmServiceImpl implements LlmService {

    @Resource
    private LlmProperties llmProperties;

    @Override
    public String model(String modelType, String system, String content) {
        if (StringUtils.isEmpty(modelType)) {
            modelType = llmProperties.getDefaultModelType();
        }
        LlmProperties.ModelConfig modelConfig = llmProperties.getModelConfig(modelType);
        if (modelType.equalsIgnoreCase("zhipu")) {
            return glmModel(modelConfig, system, content);
        } else if (modelType.equalsIgnoreCase("qwen")) {
            return qwenModel(modelConfig, system, content);
        } else if (modelType.equalsIgnoreCase("deepseek")) {
            return deepSeekModel(modelConfig, system, content);
        } else if (modelType.equalsIgnoreCase("doubao")) {
            return douBaoModel(modelConfig, system, content);
        }
        return null;
    }

    /**
     * glm模型调用
     *
     * @param modelConfig 模型配置信息
     * @param system      提示词
     * @param content     引用内容
     * @return 大模型响应结果
     */
    public String glmModel(LlmProperties.ModelConfig modelConfig, String system, String content) {
        Map<String, Object> params = new HashMap<>();
        params.put("model", modelConfig.getDefaultModel());
        List<Map<String, Object>> messageList = new ArrayList<>();
        Map<String, Object> systemMap = new HashMap<>();
        systemMap.put("role", "system");
        systemMap.put("content", system);
        messageList.add(systemMap);
        Map<String, Object> messagesMap = new HashMap<>();
        messagesMap.put("role", "user");
        messagesMap.put("content", content);
        messageList.add(messagesMap);
        params.put("messages", messageList);
        params.put("history", "[]");
        params.put("top_p", 0.7);
        //params.put("temperature", 0.75);\
        //是否开启联网搜索
        Boolean webSearchFlag = modelConfig.getWebSearch();
        if (webSearchFlag != null && webSearchFlag) {
            Map<String, Object> webSearch = new HashMap<>();
            webSearch.put("enable", true);
            webSearch.put("search_result", true);
            Map<String, Object> chatTool = new HashMap<>();
            chatTool.put("web_search", webSearch);
            chatTool.put("type", "web_search");
            List<Map<String, Object>> tools = new ArrayList<>();
            tools.add(chatTool);
            params.put("tools", tools);
        }
        String responseStr = glmPost(modelConfig.getUrl(), modelConfig.getApiKey(), new JSONObject(params));
        String choices = JSONObject.parseObject(responseStr).getString("choices");
        JSONObject choicesObject = (JSONObject) JSON.parseArray(choices).get(0);
        String messageStr = choicesObject.getString("message");
        return JSONObject.parseObject(messageStr).getString("content");
    }

    /**
     * 千问模型调用
     *
     * @param modelConfig 模型
     * @param system      提示词
     * @param content     引用内容
     * @return 大模型响应结果
     */
    public String qwenModel(LlmProperties.ModelConfig modelConfig, String system, String content) {
        try {
            Generation gen = new Generation();
            Message systemMsg = Message.builder()
                    .role(Role.SYSTEM.getValue())
                    .content(system)
                    .build();
            Message userMsg = Message.builder()
                    .role(Role.USER.getValue())
                    .content(content)
                    .build();
            GenerationParam param = GenerationParam.builder()
                    // 若没有配置环境变量，请用阿里云百炼API Key将下行替换为：.apiKey("sk-xxx")
                    .apiKey(modelConfig.getApiKey())
                    // 模型列表：https://help.aliyun.com/zh/model-studio/getting-started/models
                    .model(modelConfig.getDefaultModel())
                    .messages(Arrays.asList(systemMsg, userMsg))
                    .resultFormat(GenerationParam.ResultFormat.MESSAGE)
                    .build();
            GenerationResult call = gen.call(param);
            return call.getOutput().getChoices().get(0).getMessage().getContent();
        } catch (Exception e) {
            log.error("qwenModel error:{}", e.getMessage(), e);
        }
        return "";
    }

    public String deepSeekModel(LlmProperties.ModelConfig modelConfig, String system, String content) {
        try {
            Map<String, Object> params = new HashMap<>();
            params.put("model", "deepseek-chat");
            params.put("stream", false);
            Message systemMsg = Message.builder()
                    .role(Role.SYSTEM.getValue())
                    .content(system)
                    .build();
            Message userMsg = Message.builder()
                    .role(Role.USER.getValue())
                    .content(content)
                    .build();
            List<Message> messages = Arrays.asList(systemMsg, userMsg);
            params.put("messages", messages);
            HttpResponse execute = HttpUtil.createPost(modelConfig.getUrl())
                    .header("Content-Type", "application/json")
                    .header("Authorization", "Bearer " + modelConfig.getApiKey())
                    .body(JSONUtil.toJsonStr(params))
                    .execute();
            return execute.body();
        } catch (HttpException e) {
            log.error("deepseek error:{}", e.getMessage(), e);
        }
        return null;
    }

    public String douBaoModel(LlmProperties.ModelConfig modelConfig, String system, String content) {
        try {
            Map<String, Object> params = new HashMap<>();
            params.put("model", modelConfig.getDefaultModel());
            Message systemMsg = Message.builder()
                    .role(Role.SYSTEM.getValue())
                    .content(system)
                    .build();
            Message userMsg = Message.builder()
                    .role(Role.USER.getValue())
                    .content(content)
                    .build();
            List<Message> messages = Arrays.asList(systemMsg, userMsg);
            params.put("messages", messages);
            HttpResponse execute = HttpUtil.createPost(modelConfig.getUrl())
                    .header("Content-Type", "application/json")
                    .header("Authorization", "Bearer " + modelConfig.getApiKey())
                    .body(JSONUtil.toJsonStr(params))
                    .execute();
            String body = execute.body();
            JSONObject jsonObject = JSONObject.parseObject(body);
            JSONArray choices = jsonObject.getJSONArray("choices");
            JSONObject choice = choices.getJSONObject(0);
            JSONObject message = choice.getJSONObject("message");
            return message.getString("content");
        } catch (HttpException e) {
            log.error("doubao error:{}", e.getMessage(), e);
        }
        return null;
    }


    private String glmPost(String urlstr, String apiKey, JSONObject jsonObject) {
        String responseStr = null;
        try {
            URL url = new URL(urlstr);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod("POST");
            conn.setRequestProperty("Content-Type", "application/json; utf-8");
            conn.setRequestProperty("Accept", "application/json");
            conn.setRequestProperty("Authorization", apiKey);
            //conn.setRequestProperty("Authorization", AICHAIN_AUTHOR);
            //conn.setRequestProperty("Cookie", AICHAIN_COOKIE);
            conn.setDoOutput(true);
            String jsonInputString = jsonObject.toJSONString();
            try (OutputStream os = conn.getOutputStream()) {
                byte[] input = jsonInputString.getBytes(StandardCharsets.UTF_8);
                os.write(input, 0, input.length);
            }
            // 检查响应状态
            if (conn.getResponseCode() != HttpURLConnection.HTTP_OK) {
                throw new RuntimeException("Failed : HTTP error code : " + conn.getResponseCode());
            }
            // 读取服务器响应
            BufferedReader br = new BufferedReader(new InputStreamReader(conn.getInputStream()));
            responseStr = br.readLine();
            conn.disconnect();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return responseStr;
    }

}
