package com.zzsn.event.controller.reportData;

import cn.hutool.core.net.URLDecoder;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.google.common.base.Charsets;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.SysDictItem;
import com.zzsn.event.service.CommonService;
import com.zzsn.event.service.InformationService;
import com.zzsn.event.service.SubjectService;
import com.zzsn.event.service.SysDictItemService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.util.EsIndexUtil;
import com.zzsn.event.util.FieldUtil;
import com.zzsn.event.vo.CountVO;
import com.zzsn.event.vo.InfoDataSearchCondition;
import com.zzsn.event.vo.es.DisplayInfo;
import com.zzsn.event.vo.es.Label;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.*;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramInterval;
import org.elasticsearch.search.aggregations.bucket.histogram.Histogram;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.elasticsearch.search.aggregations.bucket.nested.NestedAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.nested.ParsedNested;
import org.elasticsearch.search.aggregations.bucket.terms.IncludeExclude;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.jsoup.Jsoup;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 专题报告数据接口
 *
 * @author lkg
 * @date 2025/4/9
 */
@Slf4j
@RestController
@RequestMapping("/report/data/api")
public class ReportDataController {

    @Resource
    private RestHighLevelClient client;
    @Autowired
    private SubjectService subjectService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private SysDictItemService dictItemService;
    @Autowired
    private InformationService informationService;

    /**
     * 检索字段集合
     *
     * @author lkg
     * @date 2025/4/9
     */
    @GetMapping("/searchFieldList")
    public Result<?> fieldList() {
        List<Map<String, Object>> fieldDetails = FieldUtil.requestFieldDetails(SearchCondition.class);
        return Result.OK(fieldDetails);
    }

    /**
     * 响应字段集合
     *
     * @author lkg
     * @date 2025/4/9
     */
    @GetMapping("/responseFieldList")
    public Result<?> responseFieldList() {
        List<Map<String, Object>> fieldDetails = FieldUtil.responseFieldDetails(ReportDataVO.class);
        return Result.OK(fieldDetails);
    }


    /**
     * 专题资讯列表(研究中心[AI资讯])
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2025/9/5
     */
    @PostMapping("/infoList")
    public Result<?> originAnalysis(@RequestBody InfoDataSearchCondition searchCondition) {
        return informationService.subjectPageListGroupByLabel(null, searchCondition);
    }

    /**
     * 专题资讯列表
     *
     * @author lkg
     * @date 2025/4/9
     */
    @PostMapping("/dataList")
    public Result<?> dataList(@RequestBody SearchCondition searchCondition) {
        try {
            List<ReportDataVO> pageList = pageList(searchCondition);
            return Result.OK(pageList);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return Result.OK();
    }

    /**
     * 词云图
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2025/4/21
     */
    @PostMapping("/wordCloud")
    public Result<?> wordCloud(@RequestBody SearchCondition searchCondition) {
        List<CountVO> list = new ArrayList<>();
        Set<String> excludeKeywords = commonService.getExcludeKeywords(searchCondition.getSubjectId());
        String[] indexArr = getIndex(searchCondition.getSubjectId());
        SearchRequest searchRequest = new SearchRequest(indexArr);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //构建查询语句
        BoolQueryBuilder boolQuery = buildQuery(searchCondition);
        searchSourceBuilder.query(boolQuery);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("groupKeywords")
                .field("keyWordsList.keyword")
                .size(searchCondition.getPageSize())
                .order(BucketOrder.count(false));
        aggregationBuilder.includeExclude((new IncludeExclude(null, String.join(",", excludeKeywords).split(","))));
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            Terms groupSource = aggregations.get("groupKeywords");
            List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String key = bucket.getKeyAsString();
                long count = bucket.getDocCount();
                CountVO countVO = new CountVO();
                countVO.setName(key);
                countVO.setValue(count);
                list.add(countVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Result.OK(list);
    }


    /**
     * 来源分析
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2025/4/21
     */
    @PostMapping("/originAnalysis")
    public Result<List<CountVO>> originAnalysis(@RequestBody SearchCondition searchCondition) {
        List<CountVO> list = new ArrayList<>();
        String[] indexArr = getIndex(searchCondition.getSubjectId());
        SearchRequest searchRequest = new SearchRequest(indexArr);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //构建查询语句
        BoolQueryBuilder boolQuery = buildQuery(searchCondition);
        searchSourceBuilder.query(boolQuery);
        //按来源分组
        TermsAggregationBuilder originAggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .size(searchCondition.getPageSize())
                .order(BucketOrder.count(false)); //返回的最大数据
        searchSourceBuilder.aggregation(originAggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            Terms groupOrigin = aggregations.get("group_origin");
            List<? extends Terms.Bucket> buckets = groupOrigin.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                CountVO countVO = new CountVO();
                countVO.setName(bucket.getKeyAsString());
                countVO.setValue(bucket.getDocCount());
                list.add(countVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Result.OK(list);
    }

    /**
     * 趋势分析(按时间分组)
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2025/4/21
     */
    @PostMapping("/trendAnalysis")
    public Result<List<CountVO>> trendAnalysis(@RequestBody SearchCondition searchCondition) {
        List<CountVO> list = new ArrayList<>();
        String[] indexArr = getIndex(searchCondition.getSubjectId());
        SearchRequest searchRequest = new SearchRequest(indexArr);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //构建查询语句
        BoolQueryBuilder boolQuery = buildQuery(searchCondition);
        searchSourceBuilder.query(boolQuery);
        //按时间分组
        DateHistogramAggregationBuilder aggregationBuilder = AggregationBuilders.dateHistogram("group_date")
                .field("publishDate")
                .calendarInterval(DateHistogramInterval.DAY)
                .format("yyyy-MM-dd");
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            ParsedDateHistogram parsedDateHistogram = aggregations.get("group_date");
            List<? extends Histogram.Bucket> buckets = parsedDateHistogram.getBuckets();
            for (Histogram.Bucket bucket : buckets) {
                CountVO countVO = new CountVO();
                countVO.setName(bucket.getKeyAsString());
                countVO.setValue(bucket.getDocCount());
                list.add(countVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Result.OK(list);
    }

    /**
     * 标签分析(按标签分组)
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    @PostMapping("/labelAnalysis")
    public Result<?> labelAnalysis(@RequestBody SearchCondition searchCondition) {
        String composeSearchLabelIds = searchCondition.getComposeSearchLabelIds();
        if (StringUtils.isBlank(composeSearchLabelIds)) {
            return Result.FAIL("请选择标签");
        }
        List<CountVO> list = new ArrayList<>();
        String[] indexArr = getIndex(searchCondition.getSubjectId());
        SearchRequest searchRequest = new SearchRequest(indexArr);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        BoolQueryBuilder boolQuery = buildQuery(searchCondition);
        searchSourceBuilder.query(boolQuery);
        String[] labelIdArr = composeSearchLabelIds.replace(";", ",").split(",");
        NestedAggregationBuilder nestedAggregationBuilder = AggregationBuilders.nested("labels", "labels")
                .subAggregation(AggregationBuilders.terms("groupLabel")
                        .field("labels.relationId")
                        .size(labelIdArr.length)
                        .includeExclude(new IncludeExclude(labelIdArr, null)));
        searchSourceBuilder.aggregation(nestedAggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            ParsedNested labels = aggregations.get("labels");
            Aggregations labelsAggregations = labels.getAggregations();
            Terms groupTag = labelsAggregations.get("groupLabel");
            List<? extends Terms.Bucket> buckets = groupTag.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String key = bucket.getKeyAsString();
                SysDictItem byId = dictItemService.getById(key);
                if (byId != null) {
                    CountVO countVO = new CountVO();
                    countVO.setName(byId.getItemText());
                    countVO.setValue(bucket.getDocCount());
                    list.add(countVO);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Result.OK(list);
    }


    /**
     * 事件对应专题库的资讯分页列表
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2024/5/6
     */
    public List<ReportDataVO> pageList(SearchCondition searchCondition) throws IOException {
        String[] indexArr = getIndex(searchCondition.getSubjectId());
        SearchRequest searchRequest = new SearchRequest(indexArr);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        String[] fetchFields = searchCondition.getFetchFields();
        String[] excludeFields = searchCondition.getExcludeFields();
        if (fetchFields != null && fetchFields.length > 0) {
            searchSourceBuilder.fetchSource(fetchFields, null);
        } else if (excludeFields != null && excludeFields.length > 0) {
            searchSourceBuilder.fetchSource(null, excludeFields);
        }
        //设置分页参数
        Integer pageNo = searchCondition.getPageNo();
        Integer pageSize = searchCondition.getPageSize();
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        searchSourceBuilder.size(pageSize);
        //排序规则
        String column = searchCondition.getColumn();
        String order = searchCondition.getOrder();
        if (searchCondition.getTopSortValid() == 1) {
            searchSourceBuilder.sort("topNum", SortOrder.DESC);
        }
        switch (column) {
            case "score":
                if (order.equals("asc")) {
                    searchSourceBuilder.sort("score", SortOrder.ASC);
                    searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                } else if (order.equals("desc")) {
                    searchSourceBuilder.sort("score", SortOrder.DESC);
                    searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                }
                break;
            case "publishDate":
                if (order.equals("desc")) {
                    searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                    searchSourceBuilder.sort("score", SortOrder.DESC);
                } else if (order.equals("asc")) {
                    searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                    searchSourceBuilder.sort("score", SortOrder.ASC);
                }
                break;
            default:
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                break;
        }
        //构建es查询条件
        BoolQueryBuilder boolQuery = buildQuery(searchCondition);
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHit[] searchHits = searchResponse.getHits().getHits();
        List<ReportDataVO> list = new ArrayList<>();
        for (SearchHit hit : searchHits) {
            String queryInfo = hit.getSourceAsString();
            ReportDataVO info = JSONUtil.toBean(queryInfo, ReportDataVO.class);
            String summary = info.getSummary();
            if (StringUtils.isNotEmpty(summary)) {
                info.setSummary(Jsoup.parse(summary).text());
            }
            info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
            info.setCreateDate(EsDateUtil.esFieldDateMapping(info.getCreateDate()));
            list.add(info);
        }
        return list;
    }

    /**
     * 构建通用的es查询语句
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2024/12/25
     */
    private BoolQueryBuilder buildQuery(SearchCondition searchCondition) {
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        List<String> ids = searchCondition.getIds();
        if (CollectionUtils.isNotEmpty(ids)) {
            boolQuery.must(QueryBuilders.termsQuery("id", ids));
        } else {
            String subjectId = searchCondition.getSubjectId();
            boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectId));
            String searchScope = searchCondition.getSearchScope();
            String searchWord = searchCondition.getSearchWord();
            if (StringUtils.isNotEmpty(searchWord)) {
                if ("all".equalsIgnoreCase(searchScope)) {
                    boolQuery.must(QueryBuilders.multiMatchQuery(searchWord, "title", "content", "origin"));
                } else {
                    boolQuery.must(QueryBuilders.matchPhraseQuery(searchScope, searchWord));
                }
            }
            //高级查询数据处理
            BoolQueryBuilder superQuery = buildSuperQuery(searchCondition.getSuperQueryMatchType(), searchCondition.getSuperQueryParams());
            if (superQuery != null) {
                boolQuery.must(superQuery);
            }
            //审核状态
            Integer status = searchCondition.getAuditStatus();
            if (status != null) {
                if (status == 1) {
                    searchCondition.setCheckStatus(1);
                } else if (status == 2) {
                    searchCondition.setCheckStatus(0);
                    searchCondition.setDeleteFlag(0);
                } else if (status == 3) {
                    searchCondition.setCheckStatus(3);
                } else if (status == 4) {
                    searchCondition.setDeleteFlag(1);
                } else if (status == 5) {
                    searchCondition.setIsFreeCheck(1);
                }
            }
            Integer checkStatus = searchCondition.getCheckStatus();
            Integer deleteFlag = searchCondition.getDeleteFlag();
            if (checkStatus != null) {
                if (checkStatus == 0) {
                    BoolQueryBuilder checkStatusBoolQuery = QueryBuilders.boolQuery();
                    checkStatusBoolQuery.should(QueryBuilders.termQuery("checkStatus", checkStatus));
                    checkStatusBoolQuery.should(QueryBuilders.boolQuery().mustNot(QueryBuilders.existsQuery("checkStatus")));
                    boolQuery.must(checkStatusBoolQuery);
                } else {
                    boolQuery.must(QueryBuilders.termQuery("checkStatus", checkStatus));
                }
            } else {
                if (deleteFlag == null) {
                    deleteFlag = 0;
                }
            }
            //删除状态查询
            if (deleteFlag != null) {
                if (deleteFlag != 0) {
                    boolQuery.must(QueryBuilders.termQuery("deleteFlag", "1"));
                } else {
                    boolQuery.mustNot(QueryBuilders.termQuery("deleteFlag", "1"));
                }
            }
            //免审核状态
            if (searchCondition.getIsFreeCheck() != null) {
                boolQuery.must(QueryBuilders.termQuery("isFreeCheck", searchCondition.getClassificationType()));
            }
            boolQuery.mustNot(QueryBuilders.matchQuery("type", "video"));
            //专题库类别筛选
            if (CollectionUtils.isNotEmpty(searchCondition.getClassificationType())) {
                boolQuery.must(QueryBuilders.termsQuery("classificationType", searchCondition.getClassificationType()));
            }
            //是否原创
            if (StringUtils.isNotEmpty(searchCondition.getOriginality())) {
                boolQuery.must(QueryBuilders.matchQuery("originality", searchCondition.getOriginality()));
            }
            //得分范围筛选
            Integer minScore = searchCondition.getMinScore();
            if (minScore != null) {
                boolQuery.filter(QueryBuilders.rangeQuery("score").gte(minScore));
            }
            Integer maxScore = searchCondition.getMaxScore();
            if (maxScore != null) {
                boolQuery.filter(QueryBuilders.rangeQuery("score").lte(maxScore));
            }
            //时间过滤筛选-前端传参
            if (searchCondition.getStartTime() != null) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(searchCondition.getStartTime()))));
            }
            if (searchCondition.getEndTime() != null) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(searchCondition.getEndTime()))));
            } else {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(new Date()))));
            }
            //组合标签查询(不同类标签之间是与的关系，同一类标签之间是或的关系)，示例: "a1,a2;c1,c2;e1,e2"
            String composeSearchLabelIds = searchCondition.getComposeSearchLabelIds();
            if (StringUtils.isNotEmpty(composeSearchLabelIds)) {
                //分号隔开的标签查询用且的关系
                String[] split = composeSearchLabelIds.split(";");
                for (String items : split) {
                    List<String> labelIds = Arrays.asList(items.split(","));
                    NestedQueryBuilder nestedQueryBuilder = QueryBuilders
                            .nestedQuery("labels", QueryBuilders.termsQuery("labels.relationId", labelIds), ScoreMode.None);
                    boolQuery.must(nestedQueryBuilder);
                }
            }
            //关联标签id集合
            List<String> labelIds = searchCondition.getLabelIds();
            if (CollectionUtils.isNotEmpty(labelIds)) {
                Set<String> relationIdSet = new HashSet<>(labelIds);
                TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", relationIdSet);
                boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            }
            //正面标签
            String tendencyLabelId = searchCondition.getTendencyLabelId();
            if (StringUtils.isNotBlank(tendencyLabelId)) {
                TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationId", tendencyLabelId);
                boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            }
            //地域标签
            List<String> areaLabelIds = searchCondition.getAreaLabelIds();
            if (CollectionUtils.isNotEmpty(areaLabelIds)) {
                Set<String> relationIdSet = new HashSet<>(areaLabelIds);
                TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", relationIdSet);
                boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            }
            //企业标签
            List<String> enterpriseLabelTypeIds = searchCondition.getEnterpriseLabelTypeIds();
            List<String> socialCreditCodeList = searchCondition.getSocialCreditCodeList();
            if (CollectionUtils.isNotEmpty(socialCreditCodeList) || CollectionUtils.isNotEmpty(enterpriseLabelTypeIds)) {
                if (CollectionUtils.isEmpty(socialCreditCodeList)) {
                    socialCreditCodeList = commonService.codesByLabels(enterpriseLabelTypeIds);
                }
                Set<String> codeSet = new HashSet<>(socialCreditCodeList);
                TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", codeSet);
                boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            }
            //信息源标签
            List<Label> labelList = searchCondition.getInfoSourceLabelList();
            if (CollectionUtils.isNotEmpty(labelList)) {
                List<String> collect = labelList.stream().map(label -> label.getLabelMark() + "-" + label.getRelationId()).collect(Collectors.toList());
                TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", collect);
                boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            }
        }
        return boolQuery;
    }

    private String[] getIndex(String subjectId) {
        List<String> subjectIdList = new ArrayList<>();
        subjectIdList.add(subjectId);
        String minDate = subjectService.getMinCreateTime(subjectIdList);
        return EsIndexUtil.getIndexIntervalYearStr(Constants.SUBJECT_INDEX, minDate);
    }

    /**
     * 高级检索条件封装
     *
     * @param superQueryMatchType 规则
     * @param superQueryParams    条件（url转码后的json）
     * @author lkg
     * @date 2025/3/13
     */
    private BoolQueryBuilder buildSuperQuery(String superQueryMatchType, String superQueryParams) {
        if (StringUtils.isNotEmpty(superQueryParams)) {
            try {
                BoolQueryBuilder superBuilder = QueryBuilders.boolQuery();
                String params = URLDecoder.decode(superQueryParams, Charsets.UTF_8);
                JSONArray array = JSON.parseArray(params);
                for (int i = 0; i < array.size(); i++) {
                    JSONObject item = array.getJSONObject(i);
                    BoolQueryBuilder builder2 = QueryBuilders.boolQuery();
                    String type = item.getString("type");
                    String rule = item.getString("rule");
                    String field = item.getString("field");
                    String val = item.getString("val");
                    if ("date".equals(type)) { //日期类型处理
                        if ("eq".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).from(val + "T00:00:00").to(val + "T23:59:59"));
                        }
                        if ("ne".equals(rule)) {
                            builder2.mustNot(QueryBuilders.rangeQuery(field).from(val + "T00:00:00").to(val + "T23:59:59"));
                        } else if ("gt".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).from((val + "T23:59:59")));
                        } else if ("ge".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).from((val + "T00:00:00")));
                        } else if ("lt".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).to((val + "T00:00:00")));
                        } else if ("le".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).to((val + "T23:59:59")));
                        }
                    } else if ("list".equals(type)) { //列表（数组）类查询
                        if ("eq".equals(rule)) {
                            builder2.must(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("ne".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("in".equals(rule)) {
                            builder2.must(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("not_in".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if (StringUtils.isNotEmpty(rule) && rule.contains("keyWordsGroup")) {
                            if (StringUtils.isNotEmpty(val)) {
                                BoolQueryBuilder wordAllBoolQuery = QueryBuilders.boolQuery();
                                String[] andWords = val.split("\\+");
                                for (String words : andWords) {
                                    String replace = words.replace("(", "").replace(")", "");
                                    BoolQueryBuilder wordOrBoolQuery = QueryBuilders.boolQuery();
                                    for (String word : replace.split("\\|")) {
                                        wordOrBoolQuery.should(QueryBuilders.termQuery(field + ".keyword", word));
                                    }
                                    wordAllBoolQuery.must(wordOrBoolQuery);
                                }
                                if(rule.startsWith("not_")){
                                    builder2.mustNot(wordAllBoolQuery);
                                }else{
                                    builder2.must(wordAllBoolQuery);
                                }
                            }
                        }
                    } else if ("string".equals(type)) { //文本类查询
                        if ("fullText".equals(field)) {
                            BoolQueryBuilder fullTextBoolQuery = multiMatchQuery(val);
                            if ("like".equals(rule)) {
                                builder2.must(fullTextBoolQuery);
                            } else if ("not_like".equals(rule)) {
                                builder2.mustNot(fullTextBoolQuery);
                            } else if (StringUtils.isNotEmpty(rule) && rule.contains("keyWordsGroup")) {
                                BoolQueryBuilder wordAllBoolQuery = QueryBuilders.boolQuery();
                                String[] andWords = val.split("\\+");
                                for (String words : andWords) {
                                    String replace = words.replace("(", "").replace(")", "");
                                    BoolQueryBuilder wordOrBoolQuery = QueryBuilders.boolQuery();
                                    for (String word : replace.split("\\|")) {
                                        wordOrBoolQuery.should(multiMatchQuery(word));
                                    }
                                    wordAllBoolQuery.must(wordOrBoolQuery);
                                }
                                if(rule.startsWith("not_")){
                                    builder2.mustNot(wordAllBoolQuery);
                                }else{
                                    builder2.must(wordAllBoolQuery);
                                }
                            }
                        } else if ("eq".equals(rule)) {
                            builder2.must(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("ne".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("in".equals(rule)) {
                            builder2.must(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("not_in".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("like".equals(rule)) {
                            builder2.must(QueryBuilders.matchPhrasePrefixQuery(field, val));
                        } else if ("not_like".equals(rule)) {
                            builder2.mustNot(QueryBuilders.matchPhrasePrefixQuery(field, val));
                        } else if (StringUtils.isNotEmpty(rule) && rule.contains("keyWordsGroup")) {
                            BoolQueryBuilder wordAllBoolQuery = QueryBuilders.boolQuery();
                            String[] andWords = val.split("\\+");
                            for (String words : andWords) {
                                String replace = words.replace("(", "").replace(")", "");
                                BoolQueryBuilder wordOrBoolQuery = QueryBuilders.boolQuery();
                                for (String word : replace.split("\\|")) {
                                    wordOrBoolQuery.should(QueryBuilders.matchPhraseQuery(field, word));
                                }
                                wordAllBoolQuery.must(wordOrBoolQuery);
                            }
                            if(rule.startsWith("not_")){
                                builder2.mustNot(wordAllBoolQuery);
                            }else{
                                builder2.must(wordAllBoolQuery);
                            }
                        }
                    }
                    if ("or".equals(superQueryMatchType)) {
                        superBuilder.should(builder2);
                    } else {
                        superBuilder.must(builder2);
                    }
                }
                return superBuilder;
            } catch (Exception e) {
                log.error("高级查询条件封装失败,e:{},params:{}", e.getMessage(), superQueryParams);
            }
        }
        return null;
    }

    private BoolQueryBuilder multiMatchQuery(String word) {
        BoolQueryBuilder fullTextBoolQuery = QueryBuilders.boolQuery();
        String[] fullTextFields = new String[]{"title", "summary", "content"};
        for (String fullTextField : fullTextFields) {
            fullTextBoolQuery.should(QueryBuilders.matchPhraseQuery(fullTextField, word));
        }
        return fullTextBoolQuery;
    }
}
